<?php

class AccountController extends LoggedUserController
{

    public function __construct($action)
    {
        parent::__construct($action);
        if (!isset($_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab'])) {
            $_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab'] = '';
            if (User::isBuyer() || User::isSigningUpBuyer()) {
                $_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab'] = 'B';
            } elseif (User::isSeller() || User::isSigningUpForSeller()) {
                $_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab'] = 'S';
            } elseif (User::isAdvertiser() || User::isSigningUpAdvertiser()) {
                $_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab'] = 'Ad';
            } elseif (User::isAffiliate() || User::isSigningUpAffiliate()) {
                $_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab'] = 'AFFILIATE';
            }
        }
        $this->set('bodyClass', 'is--dashboard');
    }

    public function index()
    {
        if (UserAuthentication::isGuestUserLogged()) {
            FatApp::redirectUser(CommonHelper::generateUrl('home'));
        }

        switch ($_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab']) {
            case 'B':
                FatApp::redirectUser(CommonHelper::generateUrl('buyer'));
                break;
            case 'S':
                FatApp::redirectUser(CommonHelper::generateUrl('seller'));
                break;
            case 'Ad':
                FatApp::redirectUser(CommonHelper::generateUrl('advertiser'));
                break;
            case 'AFFILIATE':
                FatApp::redirectUser(CommonHelper::generateUrl('affiliate'));
                break;
            default:
                FatApp::redirectUser(CommonHelper::generateUrl(''));
                break;
        }
    }

    public function viewSupplierRequest($requestId)
    {
        $requestId = FatUtility::int($requestId);

        if ($requestId < 1) {
            Message::addErrorMessage(Labels::getLabel('MSG_INVALID_REQUEST', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('Account', 'SupplierApprovalForm'));
        }

        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSupplierRequestsObj($requestId);
        $srch->addFld('tusr.*');
        $rs = $srch->getResultSet();
        $supplierRequest = FatApp::getDb()->fetch($rs);
        if (!$supplierRequest || $supplierRequest['usuprequest_id'] != $requestId) {
            Message::addErrorMessage(Labels::getLabel('MSG_INVALID_REQUEST', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('Account', 'SupplierApprovalForm'));
        }
        $maxAttempts = FatApp::getConfig('CONF_MAX_SUPPLIER_REQUEST_ATTEMPT', FatUtility::VAR_INT, 3);
        if ($supplierRequest && $supplierRequest['usuprequest_attempts'] >= $maxAttempts) {
            $this->set('maxAttemptsReached', true);
        }
        $this->set('supplierRequest', $supplierRequest);
        $this->_template->render();
    }

    public function supplierApprovalForm($p = '')
    {
        if (!User::canViewSupplierTab()) {
            Message::addErrorMessage(Labels::getLabel('MSG_INVALID_REQUEST_FOR_SUPPLIER_DASHBOARD', $this->siteLangId));
            if (User::isBuyer()) {
                FatApp::redirectUser(CommonHelper::generateUrl('buyer'));
            } elseif (User::isAdvertiser()) {
                FatApp::redirectUser(CommonHelper::generateUrl('advertiser'));
            } elseif (User::isAffiliate()) {
                FatApp::redirectUser(CommonHelper::generateUrl('affiliate'));
            } else {
                FatApp::redirectUser(CommonHelper::generateUrl('Account', 'ProfileInfo'));
            }
        }

        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSupplierRequestsObj();
        $srch->addFld(array('usuprequest_attempts', 'usuprequest_id'));

        $rs = $srch->getResultSet();
        if (!$rs) {
            Message::addErrorMessage(Labels::getLabel('MSG_INVALID_REQUEST', $this->siteLangId));
            FatUtility::dieWithError(Message::getHtml());
        }

        $supplierRequest = FatApp::getDb()->fetch($rs);
        $maxAttempts = FatApp::getConfig('CONF_MAX_SUPPLIER_REQUEST_ATTEMPT', FatUtility::VAR_INT, 3);
        if ($supplierRequest && $supplierRequest['usuprequest_attempts'] >= $maxAttempts) {
            Message::addErrorMessage(Labels::getLabel('MSG_You_have_already_consumed_max_attempts', $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('account', 'viewSupplierRequest', array($supplierRequest["usuprequest_id"])));
        }

        if ($supplierRequest && ($p != "reopen")) {
            FatApp::redirectUser(CommonHelper::generateUrl('account', 'viewSupplierRequest', array($supplierRequest["usuprequest_id"])));
        }

        $data = array('id' => isset($supplierRequest['usuprequest_id']));
        $approvalFrm = $this->getSupplierForm();
        $approvalFrm->fill($data);

        $this->set('approvalFrm', $approvalFrm);
        $this->_template->render();
    }

    public function setupSupplierApproval()
    {
        $error_messages = array();
        $fieldIdsArr = array();
        /* check if maximum attempts reached [ */
        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSupplierRequestsObj();
        $srch->addFld(array('usuprequest_attempts', 'usuprequest_id'));
        $rs = $srch->getResultSet();
        $supplierRequest = FatApp::getDb()->fetch($rs);
        /* if (!$supplierRequest) {
            $this->renderJsonError(Labels::getLabel('MSG_INVALID_REQUEST', $this->siteLangId));
        } */
        $maxAttempts = FatApp::getConfig('CONF_MAX_SUPPLIER_REQUEST_ATTEMPT', FatUtility::VAR_INT, 3);
        if (isset($supplierRequest) && $supplierRequest['usuprequest_attempts'] >= $maxAttempts) {
            $this->renderJsonError(Labels::getLabel('MSG_You_have_already_consumed_max_attempts', $this->siteLangId));
        }
        /* ] */

        $frm = $this->getSupplierForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }
        $supplier_form_fields = $userObj->getSupplierFormFields($this->siteLangId);
        foreach ($supplier_form_fields as $field) {
            $fieldIdsArr[] = $field['sformfield_id'];
            if ($field['sformfield_required'] && empty($post["sformfield_" . $field['sformfield_id']])) {
                $error_messages[] = sprintf(Labels::getLabel('MSG_Label_Required', $this->siteLangId), $field['sformfield_caption']);
            }
        }

        if (!empty($error_messages)) {
            $this->renderJsonError($error_messages);
        }
        $reference_number = $this->user["user_id"] . '-' . time();
        $data = array_merge($post, array("user_id" => $this->user["user_id"], "reference" => $reference_number, 'fieldIdsArr' => $fieldIdsArr));

        $db = FatApp::getDb();
        $db->startTransaction();
        if (!$supplier_request_id = $userObj->addSupplierRequestData($data, $this->siteLangId)) {
            $db->rollbackTransaction();
            $this->renderJsonError(Labels::getLabel('MSG_details_not_saved', $this->siteLangId));
        }

        $approval_request = 0;
        $msg = Labels::getLabel('MSG_Your_application_is_approved', $this->siteLangId);
        if (FatApp::getConfig("CONF_ADMIN_APPROVAL_SUPPLIER_REGISTRATION", FatUtility::VAR_INT, 1)) {
            $approval_request = 1;
            $msg = Labels::getLabel('MSG_Your_seller_approval_form_request_sent', $this->siteLangId);
        } 

        if (!$this->notifyAdminSupplierApproval($userObj, $data, $approval_request)) {
            $db->rollbackTransaction();
            $this->renderJsonError(Labels::getLabel("MSG_SELLER_APPROVAL_EMAIL_COULD_NOT_BE_SENT", $this->siteLangId));
        }
        $notificationData = array(
            'notification_record_type' => Notification::TYPE_USER,
            'notification_record_id' => $userObj->getMainTableRecordId(),
            'notification_user_id' => $this->user["user_id"],
            'notification_label_key' => ($approval_request) ? Notification::NEW_SUPPLIER_APPROVAL_NOTIFICATION : Notification::NEW_SELLER_APPROVED_NOTIFICATION,
            'notification_added_on' => date('Y-m-d H:i:s'),
        );

        if (!Notification::saveNotifications($notificationData)) {
            $db->rollbackTransaction();
            $this->renderJsonError(Labels::getLabel("MSG_NOTIFICATION_COULD_NOT_BE_SENT", $this->siteLangId));
        }

        $db->commitTransaction();
        $this->set('supplier_request_id', $supplier_request_id);
        $this->set('msg', $msg);
        $this->_template->render(false, false, 'json-success.php');
    }

    public function uploadSupplierFormImages()
    {
        $post = FatApp::getPostedData();
        if (empty($post)) {
            $this->renderJsonError(Labels::getLabel('LBL_Invalid_Request_Or_File_not_supported', $this->siteLangId));
        }

        $field_id = $post['field_id'];
        $fileHandlerObj = new AttachedFile();
        if (!$fileHandlerObj->isUploadedFile($_FILES['file']['tmp_name'])) {
            $this->renderJsonError($fileHandlerObj->getError());
        }

        $fileHandlerObj->deleteFile($fileHandlerObj::FILETYPE_SELLER_APPROVAL_FILE, $this->user["user_id"], 0, $field_id);
        if (!$res = $fileHandlerObj->saveAttachment($_FILES['file']['tmp_name'], $fileHandlerObj::FILETYPE_SELLER_APPROVAL_FILE, $this->user["user_id"], $field_id, $_FILES['file']['name'], -1, false)) {
            $this->renderJsonError($fileHandlerObj->getError());
        }
        $this->set('file', $_FILES['file']['name']);
        $this->set('msg', /* $_FILES['file']['name'].' '. */ Labels::getLabel('MSG_File_uploaded_successfully', $this->siteLangId));
        $this->_template->render(false, false, 'json-success.php');
    }

    public function changeEmailPassword()
    {
        $this->set('siteLangId', $this->siteLangId);
        $this->set('canSendSms', SmsArchive::canSendSms(SmsTemplate::LOGIN));
        $this->_template->render();
    }

    public function changePasswordForm()
    {
        $frm = $this->getChangePasswordForm();
        $this->set('frm', $frm);
        $this->_template->render(false, false);
    }

    public function updatePassword()
    {
        $pwdFrm = $this->getChangePasswordForm();
        $post = $pwdFrm->getFormDataFromArray(FatApp::getPostedData());
        if ($post === false) {
            $this->renderJsonError(Labels::getLabel(current($pwdFrm->getValidationErrors()), $this->siteLangId));
        }
        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSearchObj(array('user_id', 'credential_password'));
        $data = FatApp::getDb()->fetch($srch->getResultSet(), 'user_id');
        if ($data === false) {
            $this->renderJsonError(Labels::getLabel('MSG_Invalid_User', $this->siteLangId));
        }
        if ($data['credential_password'] != UserAuthentication::encryptPassword($post['current_password'])) {
            $this->renderJsonError(Labels::getLabel('MSG_YOUR_CURRENT_PASSWORD_MIS_MATCHED', $this->siteLangId));
        }
        if (!$userObj->setLoginPassword($post['new_password'])) {
            $this->renderJsonError(Labels::getLabel('MSG_Password_could_not_be_set', $this->siteLangId) . $userObj->getError());
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_Password_changed_successfully', $this->siteLangId));
    }

    public function setPrefferedDashboard($dasboardType)
    {
        $dasboardType = FatUtility::int($dasboardType);
        switch ($dasboardType) {
            case User::USER_BUYER_DASHBOARD:
                if (!User::canViewBuyerTab()) {
                    $this->renderJsonError(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
                }
                break;
            case User::USER_SELLER_DASHBOARD:
                if (!User::canViewSupplierTab()) {
                    $this->renderJsonError(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
                }
                break;
            case User::USER_ADVERTISER_DASHBOARD:
                if (!User::canViewAdvertiserTab()) {
                    $this->renderJsonError(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
                }
                break;
            case User::USER_AFFILIATE_DASHBOARD:
                if (!User::canViewAffiliateTab()) {
                    $this->renderJsonError(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
                }
                break;
            default:
                $this->renderJsonError(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
                break;
        }

        $userObj = new User($this->user["user_id"]);
        $userObj->assignValues(['user_preferred_dashboard' => $dasboardType]);
        if (!$userObj->save()) {
            $this->renderJsonError($userObj->getError());
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_Setup_successful', $this->siteLangId));
    }

    public function credits()
    {
        $frm = $this->getCreditsSearchForm($this->siteLangId);
        $canAddMoneyToWallet = true;
        if (User::isAffiliate()) {
            $canAddMoneyToWallet = false;
        }
        $codMinWalletBalance = -1;
        if (User::isSeller() && $_SESSION[UserAuthentication::SESSION_ELEMENT_NAME]['activeTab'] == 'S') {
            $shop_cod_min_wallet_balance = Shop::getAttributesByUserId($this->user["user_id"], 'shop_cod_min_wallet_balance');
            if ($shop_cod_min_wallet_balance > -1) {
                $codMinWalletBalance = $shop_cod_min_wallet_balance;
            } elseif (FatApp::getConfig('CONF_COD_MIN_WALLET_BALANCE', FatUtility::VAR_FLOAT, -1) > -1) {
                $codMinWalletBalance = FatApp::getConfig('CONF_COD_MIN_WALLET_BALANCE', FatUtility::VAR_FLOAT, -1);
            }
        }
        $txnObj = new Transactions();
        $payoutPlugins = Plugin::getNamesWithCode(Plugin::TYPE_PAYOUTS, $this->siteLangId);
        $accountSummary = $txnObj->getTransactionSummary($this->user["user_id"]);
        $payouts = [-1 => Labels::getLabel("LBL_BANK_PAYOUT", $this->siteLangId)] + $payoutPlugins;
        $this->set('payouts', $payouts);
        if (!User::isUserFromColombiaCountry($this->user["user_id"])) {
            $this->set('userWalletBalance', User::getUserBalance($this->user["user_id"]));
        } else {
            $this->set('userWalletBalance', User::getUserBalanceInColombianCurrency($this->user["user_id"]));
        }

        $this->set('codMinWalletBalance', $codMinWalletBalance);
        $this->set('frmSrch', $frm);
        $this->set('accountSummary', $accountSummary);
        $this->set('frmRechargeWallet', $this->getRechargeWalletForm($this->siteLangId));
        $this->set('canAddMoneyToWallet', $canAddMoneyToWallet);
        $this->set('sellerFromColombiaFlag', User::isUserFromColombiaCountry($this->user["user_id"]));
        $this->_template->render();
    }

    public function creditsInfo()
    {
        if (!User::isUserFromColombiaCountry($this->user["user_id"])) {
            $this->set('userWalletBalance', User::getUserBalance($this->user["user_id"]));
            $this->set('userTotalWalletBalance', User::getUserBalance($this->user["user_id"], false, false));
        } else {
            $this->set('userWalletBalance', User::getUserBalanceInColombianCurrency($this->user["user_id"]));
            $this->set('userTotalWalletBalance', User::getUserBalanceInColombianCurrency($this->user["user_id"], false, false));
        }
        $this->set('promotionWalletToBeCharged', Promotion::getPromotionWalleToBeCharged($this->user["user_id"], true));
        $this->set('withdrawlRequestAmount', User::getUserWithdrawnRequestAmount($this->user["user_id"]));
        $this->set('sellerFromColombiaFlag', User::isUserFromColombiaCountry($this->user["user_id"]));
        $this->_template->render(false, false);
    }

    public function setUpWalletRecharge()
    {
        $minimumRechargeAmount = 1;
        $frm = $this->getRechargeWalletForm($this->siteLangId);
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }
        $enteredAmount = $post['amount'];
        $loggedUserId = $this->user["user_id"];
        $moneyFormat = CommonHelper::displayMoneyFormat($minimumRechargeAmount, true, true);
        if (User::isUserFromColombiaCountry($loggedUserId)) {
            $currencyValue = 1;
            $currencyData = Currency::getCurrencyDetailWithCode(Currency::COLOMBIAN_CURRENCY_CODE);
            if (!empty($currencyData)) {
                $currencyValue = $currencyData['currency_value'];
            }
            $post['amount'] = number_format($post['amount'] / $currencyValue, 2, '.', '');
            $minimumRechargeAmount = $minimumRechargeAmount * $currencyValue;
            $moneyFormat = CommonHelper::displayMoneyFormat($minimumRechargeAmount, true, false, true, false, false, Currency::COLOMBIAN_CURRENCY_CODE);
        }

        if ($enteredAmount < $minimumRechargeAmount) {
            $str = Labels::getLabel("LBL_Recharge_amount_must_be_greater_than_{minimumrechargeamount}", $this->siteLangId);
            $this->renderJsonError(str_replace("{minimumrechargeamount}", $moneyFormat, $str));
        }

        $order_id = isset($_SESSION['wallet_recharge_cart']["order_id"]) ? $_SESSION['wallet_recharge_cart']["order_id"] : false;
        $orderData = [
            'order_type' => Orders::ORDER_WALLET_RECHARGE,
            'userAddresses' => array(),
            'order_id' => $order_id,
            'order_user_id' => $loggedUserId,
            'order_is_paid' =>  Orders::ORDER_IS_PENDING,
            'order_date_added' => date('Y-m-d H:i:s')
        ];

        /* order extras[ */
        $orderData['extra'] = array(
            'oextra_order_id' => $order_id,
            'order_ip_address' => $_SERVER['REMOTE_ADDR']
        );

        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $orderData['extra']['order_forwarded_ip'] = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } elseif (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $orderData['extra']['order_forwarded_ip'] = $_SERVER['HTTP_CLIENT_IP'];
        } else {
            $orderData['extra']['order_forwarded_ip'] = '';
        }

        $orderData['extra']['order_user_agent'] = '';
        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $orderData['extra']['order_user_agent'] = $_SERVER['HTTP_USER_AGENT'];
        }

        $orderData['extra']['order_accept_language'] = '';
        if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
            $orderData['extra']['order_accept_language'] = $_SERVER['HTTP_ACCEPT_LANGUAGE'];
        } 
        /* ] */

        $languageRow = Language::getAttributesById($this->siteLangId);
        $orderData['order_language_id'] = $languageRow['language_id'];
        $orderData['order_language_code'] = $languageRow['language_code'];

        $currencyRow = Currency::getAttributesById($this->siteCurrencyId);
        $orderData['order_currency_id'] = $currencyRow['currency_id'];
        $orderData['order_currency_code'] = $currencyRow['currency_code'];
        $orderData['order_currency_value'] = $currencyRow['currency_value'];

        $orderData['order_default_currency_to_cop'] = 1;
        $columbianCurrValue = Currency::getCurrencyDetailWithCode(Currency::COLOMBIAN_CURRENCY_CODE);
        if (!empty($columbianCurrValue)) {
            $orderData['order_default_currency_to_cop'] = $columbianCurrValue['currency_value'];
        }

        $orderData['order_user_comments'] = '';
        $orderData['order_admin_comments'] = '';
        $orderData['order_shippingapi_id'] = 0;
        $orderData['order_shippingapi_code'] = '';
        $orderData['order_tax_charged'] = 0;
        $orderData['order_site_commission'] = 0;
        $orderData['order_net_amount'] = $post['amount'];
        $orderData['order_net_amount_cop'] = $enteredAmount;
        $orderData['order_wallet_amount_charge'] = 0;
        $orderData['orderLangData'] = array();
        $orderObj = new Orders();
        if (!$orderObj->addUpdateOrder($orderData, $this->siteLangId)) {
            $this->renderJsonError($orderObj->getError());
        }
        $order_id = $orderObj->getOrderId();
        if (API_CALL) {
            /* Payment Methods[ */
            $pmSrch = PaymentMethods::getSearchObject($this->siteLangId);
            $pmSrch->doNotCalculateRecords();
            $pmSrch->doNotLimitRecords();
            $pmSrch->addMultipleFields(array('pmethod_id', 'IFNULL(pmethod_name, pmethod_identifier) as pmethod_name', 'pmethod_code', 'pmethod_description'));
            $pmSrch->addCondition('pmethod_code', '!=', 'CashOnDelivery');
            $paymentMethods = FatApp::getDb()->fetchAll($pmSrch->getResultSet());
            /* ] */
            $this->set('paymentMethods', $paymentMethods);
            $this->set('order_id', $order_id);
            $this->set('orderType', Orders::ORDER_WALLET_RECHARGE);
            $this->_template->render();
        }
        $this->set('redirectUrl', CommonHelper::generateUrl('WalletPay', 'Recharge', array($order_id)));
        $this->set('msg', Labels::getLabel('MSG_Redirecting', $this->siteLangId));
        $this->_template->render(false, false, 'json-success.php');
    }

    public function creditSearch()
    {
        $frm = $this->getCreditsSearchForm($this->siteLangId);
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        $page = FatApp::getPostedData('page', FatUtility::VAR_INT, 1);
        $pagesize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $debit_credit_type = FatApp::getPostedData('debit_credit_type', FatUtility::VAR_INT, -1);
        $dateOrder = FatApp::getPostedData('date_order', FatUtility::VAR_STRING, "DESC");
        $page = max($page, 1);
        $srch = Transactions::getUserTransactionsObj($this->user["user_id"], User::isUserFromColombiaCountry($this->user["user_id"]));
        $srch->setPageNumber($page);
        $srch->setPageSize($pagesize);
        $srch->addOrder('utxn.utxn_date', $dateOrder);
        $keyword = FatApp::getPostedData('keyword', null, '');
        if (!empty($keyword)) {
            $cond = $srch->addCondition('utxn.utxn_order_id', 'like', '%' . $keyword . '%');
            $cond->attachCondition('utxn.utxn_op_id', 'like', '%' . $keyword . '%', 'OR');
            $cond->attachCondition('utxn.utxn_comments', 'like', '%' . $keyword . '%', 'OR');
            $cond->attachCondition('concat("TN-" ,lpad( utxn.`utxn_id`,7,0))', 'like', '%' . $keyword . '%', 'OR', true);
        }

        $fromDate = FatApp::getPostedData('date_from', FatUtility::VAR_DATE, '');
        if (!empty($fromDate)) {
            $cond = $srch->addCondition('utxn.utxn_date', '>=', $fromDate);
        }

        $toDate = FatApp::getPostedData('date_to', FatUtility::VAR_DATE, '');
        if (!empty($toDate)) {
            $cond = $srch->addCondition('cast( utxn.`utxn_date` as date)', '<=', $toDate, 'and', true);
        }
        if ($debit_credit_type > 0) {
            switch ($debit_credit_type) {
                case Transactions::CREDIT_TYPE:
                    $srch->addCondition('utxn.utxn_credit', '>', '0');
                    $srch->addCondition('utxn.utxn_debit', '=', '0');
                    break;

                case Transactions::DEBIT_TYPE:
                    $srch->addCondition('utxn.utxn_debit', '>', '0');
                    $srch->addCondition('utxn.utxn_credit', '=', '0');
                    break;
            }
        }
       
        $records = array();
        $records = FatApp::getDb()->fetchAll($srch->getResultSet(), 'utxn_id');
        $this->set('arrListing', $records);
        $this->set('page', $page);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('postedData', $post);
        $this->set('statusArr', Transactions::getStatusArr($this->siteLangId));
        $this->set('sellerFromColombiaFlag', User::isUserFromColombiaCountry($this->user["user_id"]));
        if (API_CALL) {
            $this->creditsInfo();
            $this->_template->render();
        }
        $this->_template->render(false, false);
    }

    public function requestWithdrawal()
    {
        $frm = $this->getWithdrawalForm($this->siteLangId);
        if (User::isAffiliate()) {
            $fld = $frm->getField('ub_ifsc_swift_code');
            $fld->requirements()->setRegularExpressionToValidate(ValidateElement::USERNAME_REGEX);
        }
        if (User::isUserFromColombiaCountry($this->user["user_id"])) {
            $balance = User::getUserBalanceInColombianCurrency($this->user["user_id"]);
        } else {
            $balance = User::getUserBalance($this->user["user_id"]);
        }
        $lastWithdrawal = User::getUserLastWithdrawalRequest($this->user["user_id"]);
        if ($lastWithdrawal && (strtotime($lastWithdrawal["withdrawal_request_date"] . "+" . FatApp::getConfig("CONF_MIN_INTERVAL_WITHDRAW_REQUESTS", FatUtility::VAR_INT, 0) . " days") - time()) > 0) {
            $nextWithdrawalDate = date('d M,Y', strtotime($lastWithdrawal["withdrawal_request_date"] . "+" . FatApp::getConfig("CONF_MIN_INTERVAL_WITHDRAW_REQUESTS", FatUtility::VAR_INT, 0) . " days"));
            Message::addErrorMessage(sprintf(Labels::getLabel('MSG_Withdrawal_Request_Date', $this->siteLangId), FatDate::format($lastWithdrawal["withdrawal_request_date"]), FatDate::format($nextWithdrawalDate), FatApp::getConfig("CONF_MIN_INTERVAL_WITHDRAW_REQUESTS")));
            FatUtility::dieWithError(Message::getHtml());		
		}

        $minimumWithdrawLimit = FatApp::getConfig("CONF_MIN_WITHDRAW_LIMIT", FatUtility::VAR_INT, 0);
        $moneyFormat = CommonHelper::displayMoneyFormat($minimumWithdrawLimit);
        if (User::isUserFromColombiaCountry($this->user["user_id"])) {
            $currencyValue = 1;
            $currencyData = Currency::getCurrencyDetailWithCode(Currency::COLOMBIAN_CURRENCY_CODE);
            if (!empty($currencyData)) {
                $currencyValue = $currencyData['currency_value'];
            }
            $minimumWithdrawLimit = $minimumWithdrawLimit * $currencyValue;
            $moneyFormat = CommonHelper::displayMoneyFormat($minimumWithdrawLimit, true, false, true, false, false, Currency::COLOMBIAN_CURRENCY_CODE);
        }
        if ($balance < $minimumWithdrawLimit) {
            Message::addErrorMessage(sprintf(Labels::getLabel('MSG_Withdrawal_Request_Minimum_Balance_Less', $this->siteLangId), CommonHelper::displayMoneyFormat($minimumWithdrawLimit)));
            FatUtility::dieWithError(Message::getHtml());			
        }

        $userObj = new User($this->user["user_id"]);
        $data = $userObj->getUserBankInfo();
        $data['uextra_payment_method'] = User::AFFILIATE_PAYMENT_METHOD_CHEQUE;
        if (User::isAffiliate()) {
            $userExtraData = User::getUserExtraData($this->user["user_id"], array('uextra_payment_method', 'uextra_cheque_payee_name', 'uextra_paypal_email_id'));
            $uextra_payment_method = isset($userExtraData['uextra_payment_method']) ? $userExtraData['uextra_payment_method'] : User::AFFILIATE_PAYMENT_METHOD_CHEQUE;
            $data = array_merge($data, $userExtraData);
            $data['uextra_payment_method'] = $uextra_payment_method;
            $this->set('uextra_payment_method', $uextra_payment_method);
        }
        $frm->fill($data);
        $this->set('frm', $frm);
        $this->_template->render(false, false);
    }

    public function setupRequestWithdrawal()
    {
        if (User::isUserFromColombiaCountry($this->user["user_id"])) {
            $balance = User::getUserBalanceInColombianCurrency($this->user["user_id"]);
        } else {
            $balance = User::getUserBalance($this->user["user_id"]);
        }
        $lastWithdrawal = User::getUserLastWithdrawalRequest($this->user["user_id"]);
        if ($lastWithdrawal && (strtotime($lastWithdrawal["withdrawal_request_date"] . "+" . FatApp::getConfig("CONF_MIN_INTERVAL_WITHDRAW_REQUESTS", FatUtility::VAR_INT, 0) . " days") - time()) > 0) {
            $nextWithdrawalDate = date('d M,Y', strtotime($lastWithdrawal["withdrawal_request_date"] . "+" . FatApp::getConfig("CONF_MIN_INTERVAL_WITHDRAW_REQUESTS") . " days"));
            $message = sprintf(Labels::getLabel('MSG_Withdrawal_Request_Date', $this->siteLangId), FatDate::format($lastWithdrawal["withdrawal_request_date"]), FatDate::format($nextWithdrawalDate), FatApp::getConfig("CONF_MIN_INTERVAL_WITHDRAW_REQUESTS"));
            $this->renderJsonError($message);
        }

        $minimumWithdrawLimit = FatApp::getConfig("CONF_MIN_WITHDRAW_LIMIT", FatUtility::VAR_INT, 0);
        $maximumWithdrawLimit = FatApp::getConfig("CONF_MAX_WITHDRAW_LIMIT", FatUtility::VAR_INT, 0);
        $moneyFormat = CommonHelper::displayMoneyFormat($minimumWithdrawLimit);
        $currencyValue = 1;
        if (User::isUserFromColombiaCountry($this->user["user_id"])) {
            $currencyData = Currency::getCurrencyDetailWithCode(Currency::COLOMBIAN_CURRENCY_CODE);
            if (!empty($currencyData)) {
                $currencyValue = $currencyData['currency_value'];
            }
            $minimumWithdrawLimit = $minimumWithdrawLimit * $currencyValue;
            $maximumWithdrawLimit = $maximumWithdrawLimit * $currencyValue;
            $moneyFormat = CommonHelper::displayMoneyFormat($minimumWithdrawLimit, true, false, true, false, false, Currency::COLOMBIAN_CURRENCY_CODE);
        }
        if ($balance < $minimumWithdrawLimit) {
            $this->renderJsonError(sprintf(Labels::getLabel('MSG_Withdrawal_Request_Minimum_Balance_Less', $this->siteLangId), $moneyFormat));
        }
        $frm = $this->getWithdrawalForm($this->siteLangId);
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }

        if (($minimumWithdrawLimit > $post["withdrawal_amount"])) {
            $message = sprintf(Labels::getLabel('MSG_Your_withdrawal_request_amount_is_less_than_the_minimum_allowed_amount_of_%s', $this->siteLangId), $moneyFormat);
            $this->renderJsonError($message);
        }

        if ($maximumWithdrawLimit != 0 && ($maximumWithdrawLimit < $post["withdrawal_amount"])) {
            $message = sprintf(Labels::getLabel('MSG_Your_withdrawal_request_amount_is_greater_than_the_maximum_allowed_amount_of_%s', $this->siteLangId), $moneyFormat);
            $this->renderJsonError($message);
        }

        if (($post["withdrawal_amount"] > $balance)) {
            $message = Labels::getLabel('MSG_Withdrawal_Request_Greater', $this->siteLangId);
            $this->renderJsonError($message);
        }

        $accountNumber = FatApp::getPostedData('ub_account_number', FatUtility::VAR_STRING, 0);
        if ((string) $accountNumber != $post['ub_account_number']) {
            $message = Labels::getLabel('MSG_Invalid_Account_Number', $this->siteLangId);
            $this->renderJsonError($message);
        }

        $userObj = new User($this->user["user_id"]);
        if (!$userObj->updateBankInfo($post)) {
            $this->renderJsonError(Labels::getLabel($userObj->getError(), $this->siteLangId));
        }

        $withdrawal_payment_method = FatApp::getPostedData('uextra_payment_method', FatUtility::VAR_INT, 0);
        $withdrawal_payment_method = ($withdrawal_payment_method > 0 && array_key_exists($withdrawal_payment_method, User::getAffiliatePaymentMethodArr($this->siteLangId))) ? $withdrawal_payment_method : User::AFFILIATE_PAYMENT_METHOD_BANK;
        $withdrawal_cheque_payee_name = '';
        $withdrawal_paypal_email_id = '';
        $withdrawal_bank = '';
        $withdrawal_account_holder_name = '';
        $withdrawal_account_number = '';
        $withdrawal_ifc_swift_code = '';
        $withdrawal_bank_address = '';
        $withdrawal_comments = $post['withdrawal_comments'];

        switch ($withdrawal_payment_method) {
            case User::AFFILIATE_PAYMENT_METHOD_CHEQUE:
                $withdrawal_cheque_payee_name = $post['uextra_cheque_payee_name'];
                break;
            case User::AFFILIATE_PAYMENT_METHOD_BANK:
                $withdrawal_bank = $post['ub_bank_name'];
                $withdrawal_account_holder_name = $post['ub_account_holder_name'];
                $withdrawal_account_number = $post['ub_account_number'];
                $withdrawal_ifc_swift_code = $post['ub_ifsc_swift_code'];
                $withdrawal_bank_address = $post['ub_bank_address'];
                break;
            case User::AFFILIATE_PAYMENT_METHOD_PAYPAL:
                $withdrawal_paypal_email_id = $post['uextra_paypal_email_id'];
                break;
        }


        $post['withdrawal_payment_method'] = $withdrawal_payment_method;
        $post['withdrawal_cheque_payee_name'] = $withdrawal_cheque_payee_name;
        $post['withdrawal_paypal_email_id'] = $withdrawal_paypal_email_id;

        $post['ub_bank_name'] = $withdrawal_bank;
        $post['ub_account_holder_name'] = $withdrawal_account_holder_name;
        $post['ub_account_number'] = $withdrawal_account_number;
        $post['ub_ifsc_swift_code'] = $withdrawal_ifc_swift_code;
        $post['ub_bank_address'] = $withdrawal_bank_address;

        $post['withdrawal_comments'] = $withdrawal_comments;
        $post["withdrawal_original_amount"] = $post["withdrawal_amount"];
        $post["withdrawal_user_from_colombia"] = 0;
        $currencyValue = 1;
        if (User::isUserFromColombiaCountry($this->user["user_id"])) {
            $currencyData = Currency::getCurrencyDetailWithCode(Currency::COLOMBIAN_CURRENCY_CODE);
            if (!empty($currencyData)) {
                $currencyValue = $currencyData['currency_value'];
                $post["withdrawal_amount"] = number_format($post["withdrawal_amount"] / $currencyValue, 4, '.', '');
            }
            $post["withdrawal_user_from_colombia"] = 1;
        }
        $post['withdrawal_currency_conversion_rate'] = $currencyValue;

        if (!$withdrawRequestId = $userObj->addWithdrawalRequest(array_merge($post, array("ub_user_id" => $this->user["user_id"])), $this->siteLangId, User::isSeller())) {
            $message = Labels::getLabel($userObj->getError(), $this->siteLangId);
            $this->renderJsonError($message);
        }

        $emailNotificationObj = new EmailHandler();
        if (!$emailNotificationObj->sendWithdrawRequestNotification($withdrawRequestId, $this->siteLangId, "A")) {
            $message = Labels::getLabel($emailNotificationObj->getError(), $this->siteLangId);
            $this->renderJsonError($message);
        }
        
        //send notification to admin
        $notificationData = array(
            'notification_record_type' => Notification::TYPE_WITHDRAWAL_REQUEST,
            'notification_record_id' => $withdrawRequestId,
            'notification_user_id' => $this->user["user_id"],
            'notification_label_key' => Notification::WITHDRAWL_REQUEST_NOTIFICATION,
            'notification_added_on' => date('Y-m-d H:i:s'),
        );
        
        if (!Notification::saveNotifications($notificationData)) {
            $this->renderJsonError(Labels::getLabel("MSG_NOTIFICATION_COULD_NOT_BE_SENT", $this->siteLangId));
        }

        $this->renderJsonSuccess(Labels::getLabel('MSG_Withdraw_request_placed_successfully', $this->siteLangId));
    }

    public function removeProfileImage()
    {
        $fileHandlerObj = new AttachedFile();
        if (!$fileHandlerObj->deleteFile(AttachedFile::FILETYPE_USER_PROFILE_IMAGE, $this->user["user_id"])) {
            $this->renderJsonError(Labels::getLabel($fileHandlerObj->getError(), $this->siteLangId));
        }

        if (!$fileHandlerObj->deleteFile(AttachedFile::FILETYPE_USER_PROFILE_CROPED_IMAGE, $this->user["user_id"])) {
            $this->renderJsonError(Labels::getLabel($fileHandlerObj->getError(), $this->siteLangId));
        }
        if (API_CALL) {
            $userImgUpdatedOn = User::getAttributesById($this->user["user_id"], 'user_updated_on');
            $uploadedTime = AttachedFile::setTimeParam($userImgUpdatedOn);
            $userImage = FatCache::getCachedUrl(CommonHelper::generateFullFileUrl('image', 'user', array($this->user["user_id"], 'thumb', true)) . $uploadedTime, CONF_IMG_CACHE_TIME, '.jpg');
            $this->renderJsonSuccess(Labels::getLabel('MSG_Profile_Image_Removed_Successfully', $this->siteLangId), ['data' => ['userImage' => $userImage]]);
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_Profile_Image_Removed_Successfully', $this->siteLangId));
    }

    public function userProfileImage($userId, $sizeType = '', $cropedImage = false)
    {
        $default_image = 'user_deafult_image.jpg';
        $recordId = FatUtility::int($this->user["user_id"]);

        $file_row = false;
        if ($cropedImage == true) {
            $file_row = AttachedFile::getAttachment(AttachedFile::FILETYPE_USER_PROFILE_CROPED_IMAGE, $recordId);
        }

        if ($file_row == false) {
            $file_row = AttachedFile::getAttachment(AttachedFile::FILETYPE_USER_PROFILE_IMAGE, $recordId);
        }

        $image_name = isset($file_row['afile_physical_path']) ? $file_row['afile_physical_path'] : '';
        switch (strtoupper($sizeType)) {
            case 'THUMB':
                $w = 100;
                $h = 100;
                AttachedFile::displayImage($image_name, $w, $h, $default_image);
                break;
            default:
                AttachedFile::displayOriginalImage($image_name, $default_image);
                break;
        }
    }

    public function profileInfo()
    {
        if (API_CALL) {
            $userImgUpdatedOn = User::getAttributesById($this->user["user_id"], 'user_updated_on');
            $uploadedTime = AttachedFile::setTimeParam($userImgUpdatedOn);
            $hasDigitalProducts = 0;
            $srch = Product::getSearchObject();
            $srch->addMultipleFields(['product_id']);
            $srch->addCondition('product_type', '=', Product::PRODUCT_TYPE_DIGITAL);
            $srch->setPageSize(1);
            $rs = $srch->getResultSet();
            $row = $this->db->fetch($rs);
            if (!empty($row) && 0 < count($row)) {
                $hasDigitalProducts = 1;
            }

            $bankInfo = $this->bankInfo(true);
            $personalInfo = $this->personalInfo(true);
            $personalInfo['userImage'] = FatCache::getCachedUrl(CommonHelper::generateFullFileUrl('image', 'user', array($this->user['user_id'], 'SMALL', true)) . $uploadedTime, CONF_IMG_CACHE_TIME, '.jpg');
            $this->set('personalInfo', empty($personalInfo) ? (object) array() : $personalInfo);
            $this->set('bankInfo', empty($bankInfo) ? (object) array() : $bankInfo);
            $this->set('privacyPolicyLink', FatApp::getConfig('CONF_PRIVACY_POLICY_PAGE', FatUtility::VAR_STRING, ''));
            $this->set('hasDigitalProducts', $hasDigitalProducts);
            $this->_template->render();
            exit;
        }

        $this->_template->addJs('js/jquery.form.js');
        $this->_template->addJs('js/cropper.js');
        $this->_template->addJs('js/cropper-main.js');
        $this->includeDateTimeFiles();
        $showSellerActivateButton = false;
        if (!User::canAccessSupplierDashboard() && $this->user['user_registered_initially_for'] == User::USER_TYPE_SELLER) {
            $showSellerActivateButton = true;
        }
        $payoutPlugins = Plugin::getNamesWithCode(Plugin::TYPE_PAYOUTS, $this->siteLangId);
        $this->set('userParentId', $this->user['user_parent']);
        $this->set('payouts', $payoutPlugins);
        $this->set('showSellerActivateButton', $showSellerActivateButton);
        $this->set('userPreferredDashboard', $this->user['user_preferred_dashboard']);
        $this->_template->render();
    }

    public function personalInfo($return = false)
    {
        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSearchObj();
        $srch->addMultipleFields(array('u.*', 'country_name', 'state_name'));
        $srch->joinTable('tbl_countries_lang', 'LEFT JOIN', 'countrylang_country_id = user_country_id and countrylang_lang_id = ' . $this->siteLangId);
        $srch->joinTable('tbl_states_lang', 'LEFT JOIN', 'statelang_state_id = user_state_id and statelang_lang_id = ' . $this->siteLangId);
        $data = FatApp::getDb()->fetch($srch->getResultSet(), 'user_id');
        if ($return) {
            return $data;
        }
        $this->set('info', $data);
        $this->_template->render(false, false);
    }

    public function bankInfo($return = false)
    {
        $userObj = new User($this->user["user_id"]);
        $data = $userObj->getUserBankInfo();
        if ($return) {
            return $data;
        }
        $this->set('info', $data);
        $this->_template->render(false, false);
    }

    public function profileInfoForm()
    {
        $frm = $this->getProfileInfoForm();
        $imgFrm = $this->getProfileImageForm();
        $stateId = 0;
        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSearchObj();
        $srch->addMultipleFields(array('u.*'));
        $data = FatApp::getDb()->fetch($srch->getResultSet(), 'user_id');
        if (User::isAffiliate()) {
            $userExtraData = User::getUserExtraData($this->user["user_id"], array('uextra_company_name', 'uextra_website'));
            $userExtraData = ($userExtraData) ? $userExtraData : array();
            $data = array_merge($userExtraData, $data);
        }
         
        if ($data['user_dob'] == "0000-00-00") {
            $dobFld = $frm->getField('user_dob');
            $dobFld->requirements()->setRequired(true);
        }

        $frm->fill($data);
        $stateId = $data['user_state_id'];
        $mode = 'Add';
        $file_row = AttachedFile::getAttachment(AttachedFile::FILETYPE_USER_PROFILE_IMAGE, $this->user["user_id"]);
        if ($file_row != false) {
            $mode = 'Edit';
        }
        $this->set('countryIso', User::getUserMeta($this->user["user_id"], 'user_country_iso'));
        $this->set('data', $data);
        $this->set('frm', $frm);
        $this->set('imgFrm', $imgFrm);
        $this->set('mode', $mode);
        $this->set('stateId', $stateId);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function imgCropper()
    {
        $userImgUpdatedOn = User::getAttributesById($this->user["user_id"], 'user_updated_on');
        $uploadedTime = AttachedFile::setTimeParam($userImgUpdatedOn);
        $userImage = FatCache::getCachedUrl(CommonHelper::generateFullFileUrl('image', 'user', [$this->user["user_id"]]) . $uploadedTime, CONF_IMG_CACHE_TIME, '.jpg');
        $this->set('image', $userImage);
        $this->_template->render(false, false, 'cropper/index.php');
    }

    public function profileImageForm()
    {
        $imgFrm = $this->getProfileImageForm();
        $mode = 'Add';
        $file_row = AttachedFile::getAttachment(AttachedFile::FILETYPE_USER_PROFILE_IMAGE, $this->user["user_id"]);
        if ($file_row != false) {
            $mode = 'Edit';
        }
        $this->set('mode', $mode);
        $this->set('imgFrm', $imgFrm);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function uploadProfileImage()
    {
        $post = FatApp::getPostedData();
        if (empty($post)) {
            $this->renderJsonError(Labels::getLabel('LBL_Invalid_Request_Or_File_not_supported', $this->siteLangId));
        }
        $updatedAt = date('Y-m-d H:i:s');
        $uploadedTime = AttachedFile::setTimeParam($updatedAt);
        if (isset($_FILES['org_image']['tmp_name'])) {
            $fileHandlerObj = new AttachedFile();
            if (!$fileHandlerObj->isUploadedFile($_FILES['org_image']['tmp_name'])) {
                $this->renderJsonError($fileHandlerObj->getError());
            }

            if (!$fileHandlerObj->saveImage($_FILES['org_image']['tmp_name'], AttachedFile::FILETYPE_USER_PROFILE_IMAGE, $this->user["user_id"], 0, $_FILES['org_image']['name'], -1, true)) {
                $this->renderJsonError(Labels::getLabel($fileHandlerObj->getError(), $this->siteLangId));
            }
        }

        if (isset($_FILES['cropped_image']['tmp_name'])) {
            $fileHandlerObj = new AttachedFile();
            if (!$fileHandlerObj->isUploadedFile($_FILES['cropped_image']['tmp_name'])) {
                $this->renderJsonError($fileHandlerObj->getError());
            }

            if (!$fileHandlerObj->saveImage($_FILES['cropped_image']['tmp_name'], AttachedFile::FILETYPE_USER_PROFILE_CROPED_IMAGE, $this->user["user_id"], 0, $_FILES['cropped_image']['name'], -1, true)) {
                $this->renderJsonError(Labels::getLabel($fileHandlerObj->getError(), $this->siteLangId));
            }
        }

        $url = CommonHelper::generateFullFileUrl('Account', 'userProfileImage', array($this->user["user_id"], 'croped', true)) . $uploadedTime;
        if (API_CALL) {
            $url = CommonHelper::generateFullFileUrl('image', 'user', array($this->user["user_id"], 'mini', true)) . $uploadedTime;
        }
        $file =  FatCache::getCachedUrl($url, CONF_IMG_CACHE_TIME, '.jpg');
        User::setImageUpdatedOn($this->user["user_id"], $updatedAt);
        $this->renderJsonSuccess(Labels::getLabel('MSG_File_uploaded_successfully', $this->siteLangId), ["data" => ["file" => $file]]);
    }

    public function updateProfileInfo()
    {
        $frm = $this->getProfileInfoForm();
        $post = FatApp::getPostedData();
        if (API_CALL && 1 > count($post)) {
            $this->renderJsonError(Labels::getLabel("MSG_INVALID_REQUEST", $this->siteLangId));
        }
        /* CommonHelper::printArray($post);  */
        $user_state_id = FatUtility::int($post['user_state_id']);
        $post = $frm->getFormDataFromArray($post);
        if (false === $post) {
            $this->renderJsonError(Labels::getLabel(current($frm->getValidationErrors()), $this->siteLangId));
        }
        if (strtotime($post['user_dob']) > time()) {
            $this->renderJsonError(Labels::getLabel("MSG_Invalid_date_of_birth", $this->siteLangId));
        }
        $post['user_state_id'] = $user_state_id;
        if (isset($post['user_id'])) {
            unset($post['user_id']);
        }
        if (isset($post['user_phone']) && true == SmsArchive::canSendSms()) {
            unset($post['user_phone']);
        }
        if ($post['user_dob'] == "0000-00-00" || $post['user_dob'] == "" || strtotime($post['user_dob']) == 0) {
            unset($post['user_dob']);
        }
        unset($post['credential_username'], $post['credential_email']);
        /* saving user extras[ */
        if (User::isAffiliate()) {
            $dataToSave = array(
                'uextra_user_id' => $this->user["user_id"],
                'uextra_company_name' => $post['uextra_company_name'],
                'uextra_website' => CommonHelper::processUrlString($post['uextra_website'])
            );
            $dataToUpdateOnDuplicate = $dataToSave;
            unset($dataToUpdateOnDuplicate['uextra_user_id']);
            if (!FatApp::getDb()->insertFromArray(User::DB_TBL_USR_EXTRAS, $dataToSave, false, array(), $dataToUpdateOnDuplicate)) {
                $message = Labels::getLabel("LBL_Details_could_not_be_saved!", $this->siteLangId);
                if (API_CALL) {
                    $this->renderJsonError($message);
                }

                Message::addErrorMessage($message);
                if (FatUtility::isAjaxCall()) {
                    FatUtility::dieWithError(Message::getHtml());
                }
                FatApp::redirectUser(CommonHelper::generateUrl('Account', 'ProfileInfo'));
            }
        }
        /* ] */

        if (false == SmsArchive::canSendSms()) {
            $post['user_dial_code'] = FatApp::getPostedData('user_dial_code', FatUtility::VAR_STRING, '');
            $countryIso = FatApp::getPostedData('user_country_iso', FatUtility::VAR_STRING, '');
        }

        $userObj = new User($this->user["user_id"]);
        $userObj->assignValues($post);
        if (!$userObj->save()) {
            $message = Labels::getLabel($userObj->getError(), $this->siteLangId);
            $this->renderJsonError($message);
        }

        if (false == SmsArchive::canSendSms()) {
            $user = clone $userObj;
            if (false === $user->updateUserMeta('user_country_iso', $countryIso)) {
                $this->renderJsonError($user->getError());
            }
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_Updated_Successfully', $this->siteLangId));
    }

    public function bankInfoForm()
    {
        if (User::isAffiliate()) {
            Message::addErrorMessage(Labels::getLabel('LBL_Invalid_Request', $this->siteLangId));
            FatUtility::dieWithError(Message::getHtml());
        }
        $frm = $this->getBankInfoForm();
        $userObj = new User($this->user["user_id"]);
        $data = $userObj->getUserBankInfo();
        if ($data != false) {
            $frm->fill($data);
        }
        $this->set('frm', $frm);
        $this->_template->render(false, false);
    }

    public function settingsInfo()
    {
        $frm = $this->getSettingsForm();
        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSearchObj();
        $srch->addMultipleFields(array('u.*'));
        $rs = $srch->getResultSet();
        $data = FatApp::getDb()->fetch($rs, 'user_id');
        if ($data != false) {
            $frm->fill($data);
        }
        $this->set('frm', $frm);
        $this->_template->render(false, false);
    }

    public function updateBankInfo()
    {
        $post = FatApp::getPostedData();
        if (API_CALL && 1 > count($post)) {
            $this->renderJsonError(Labels::getLabel("MSG_INVALID_REQUEST", $this->siteLangId));
        }
        $frm = $this->getBankInfoForm();
        $post = $frm->getFormDataFromArray($post);
        if (false === $post) {
            $this->renderJsonError(Labels::getLabel(current($frm->getValidationErrors()), $this->siteLangId));
        }
        $accountNumber = FatApp::getPostedData('ub_account_number', FatUtility::VAR_STRING, 0);
        if ((string) $accountNumber != $post['ub_account_number']) {
            $this->renderJsonError(Labels::getLabel('MSG_Invalid_Account_Number', $this->siteLangId));
        }
        $userObj = new User($this->user["user_id"]);
        if (!$userObj->updateBankInfo($post)) {
            $this->renderJsonError(Labels::getLabel($userObj->getError(), $this->siteLangId));
        }

        $this->renderJsonSuccess(Labels::getLabel('MSG_Updated_Successfully', $this->siteLangId));
    }

    public function updateSettingsInfo()
    {
        $frm = $this->getSettingsForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }

        $userObj = new User($this->user["user_id"]);
        if (!$userObj->updateSettingsInfo($post)) {
            $this->renderJsonError(Labels::getLabel($userObj->getError(), $this->siteLangId));
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_Setup_successful', $this->siteLangId));
    }

    public function changeEmailForm()
    {
        $frm = $this->getChangeEmailForm();

        $this->set('frm', $frm);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function updateEmail()
    {
        $emailFrm = $this->getChangeEmailForm();
        $post = $emailFrm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($emailFrm->getValidationErrors()));
        }
        
        if ($post['new_email'] != $post['conf_new_email']) {
            $this->renderJsonError(Labels::getLabel('MSG_New_email_confirm_email_does_not_match', $this->siteLangId));
        }
        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSearchObj(array('user_id', 'credential_password'));
        $data = FatApp::getDb()->fetch($srch->getResultSet(), 'user_id');
        $existingPassword  = $data['credential_password'];
        if ($existingPassword != UserAuthentication::encryptPassword($post['current_password'])) {
            $this->renderJsonError(Labels::getLabel('MSG_YOUR_CURRENT_PASSWORD_MIS_MATCHED', $this->siteLangId));
        }

        $phone = !empty($this->user['user_phone']) ? $this->user['user_dial_code'] . $this->user['user_phone'] : '';
        $arr = array(
            'user_name' => $this->user['user_name'],
            'user_phone' => $phone,
            'user_email' => $this->user['credential_email'],
            'user_new_email' => $post['new_email']
        );
        $userObj = new User($this->user["user_id"]);
        if (!$this->userEmailVerifications($userObj, $arr)) {
            $this->renderJsonError(Labels::getLabel('MSG_ERROR_IN_SENDING_VERFICATION_EMAIL', $this->siteLangId));
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_CHANGE_EMAIL_REQUEST_SENT_SUCCESSFULLY', $this->siteLangId));
    }

    public function moveToWishList($selProdId)
    {
        $defaultWishListId = $this->getDefaultWishListId();
        $this->addRemoveWishListProduct($selProdId, $defaultWishListId);
    }

    public function getDefaultWishListId()
    {
        $loggedUserId = $this->user["user_id"];
        $srch = UserWishList::getSearchObject($loggedUserId, true);
        $srch->addCondition('uwlist_default', '=', AppConstants::YES);
        $srch->addMultipleFields(array('uwlist_id'));
        $srch->setPageSize(1);
        $rs = $srch->getResultSet();
        $row = FatApp::getDb()->fetch($rs);

        if (!empty($row)) {
            return $row['uwlist_id'];
        }

        $wListObj = new UserWishList();
        $data_to_save_arr['uwlist_title'] = Labels::getLabel('LBL_Default_list', $this->siteLangId);;
        $data_to_save_arr['uwlist_added_on'] = date('Y-m-d H:i:s');
        $data_to_save_arr['uwlist_user_id'] = $loggedUserId;
        $data_to_save_arr['uwlist_default'] = 1;
        $wListObj->assignValues($data_to_save_arr);

        /* create new List[ */
        if (!$wListObj->save()) {
            Message::addErrorMessage($wListObj->getError());
            FatUtility::dieWithError(Message::getHtml());
        }
        $uwlist_id = $wListObj->getMainTableRecordId();
        return $uwlist_id;
    }

    /* called from products listing page */

    public function viewWishList($selprod_id, $excludeWishList = 0)
    {
        $excludeWishList = FatUtility::int($excludeWishList);
        $loggedUserId = $this->user["user_id"];
        $this->getDefaultWishListId();
        $wishLists = UserWishList::getUserWishLists($loggedUserId, true, $excludeWishList);
        $frm = $this->getCreateWishListForm();
        $frm->fill(array('selprod_id' => $selprod_id));
        $this->set('frm', $frm);
        $this->set('wishLists', $wishLists);
        $this->set('selprod_id', $selprod_id);
        $this->_template->render(false, false);
    }

    public function setupWishList()
    {
        $frm = $this->getCreateWishListForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        $selprod_id = FatUtility::int($post['selprod_id']);
        if (false === $post) {
            $message = current($frm->getValidationErrors());
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }
        $loggedUserId = $this->user["user_id"];
        $wListObj = new UserWishList();
        $data_to_save_arr = $post;
        $data_to_save_arr['uwlist_added_on'] = date('Y-m-d H:i:s');
        $data_to_save_arr['uwlist_user_id'] = $this->user["user_id"];
        $wListObj->assignValues($data_to_save_arr);

        /* create new List[ */
        if (!$wListObj->save()) {
            $message = $wListObj->getError();
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }
        $uwlp_uwlist_id = $wListObj->getMainTableRecordId();
        /* ] */

        $successMsg = Labels::getLabel('LBL_WishList_Created_Successfully', $this->siteLangId);
        /* Assign current product to newly created list[ */
        if ($uwlp_uwlist_id && $selprod_id) {
            if (!$wListObj->addUpdateListProducts($uwlp_uwlist_id, $selprod_id)) {
                Message::addMessage($successMsg);
                $msg = Labels::getLabel('LBL_Error_while_assigning_product_under_selected_list.', $this->siteLangId);

                if (API_CALL) {
                    LibHelper::dieJsonError($msg);
                }
                Message::addErrorMessage($msg);
                FatUtility::dieWithError(Message::getHtml());
            }
        }
        /* ] */

        //UserWishList
        $srch = UserWishList::getSearchObject($loggedUserId);
        $srch->joinTable(UserWishList::DB_TBL_LIST_PRODUCTS, 'LEFT OUTER JOIN', 'uwlist_id = uwlp_uwlist_id');
        $srch->addCondition('uwlp_selprod_id', '=', $selprod_id);
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addMultipleFields(array('uwlist_id'));
        $rs = $srch->getResultSet();
        $row = FatApp::getDb()->fetch($rs);
        $productIsInAnyList = false;
        if ($row) {
            $productIsInAnyList = true;
        }

        $this->set('productIsInAnyList', $productIsInAnyList);
        $this->set('wish_list_id', $uwlp_uwlist_id);
        $this->set('msg', $successMsg);
        if (API_CALL) {
            $this->_template->render();
        }
        $this->_template->render(false, false, 'json-success.php');
    }

    public function addRemoveWishListProductArr()
    {
        $selprod_id_arr = FatApp::getPostedData('selprod_id');
        $selprod_id_arr = !empty($selprod_id_arr) ? array_filter($selprod_id_arr) : array();

        $uwlist_id = FatApp::getPostedData('uwlist_id', FatUtility::VAR_INT, 0);

        if (empty($selprod_id_arr) || empty($uwlist_id)) {
            $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        foreach ($selprod_id_arr as $selprod_id) {
            $action = $this->updateWishList($selprod_id, $uwlist_id);
        }

        if (API_CALL) {
            $this->_template->render();
        }

        $this->_template->render(false, false, 'json-success.php');
    }

    public function updateRemoveWishListProduct($selprodId, $wishListId)
    {
        $selprodIdArr = FatApp::getPostedData('selprod_id');
        $oldWishlistId = FatApp::getPostedData('uwlist_id', FatUtility::VAR_INT, 0);

        if (empty($selprodIdArr) || empty($oldWishlistId)) {
            Message::addErrorMessage(Labels::getLabel("LBL_Invalid_Request", $this->siteLangId));
            FatUtility::dieWithError(Message::getHtml());
        }
        foreach ($selprodIdArr as $selprodId) {
            $this->updateWishList($selprodId, $oldWishlistId);
            $isExists = UserWishList::getListProductsByListId($wishListId, $selprodId);
            if (empty($isExists)) {
                $this->updateWishList($selprodId, $wishListId);
            }
        }
        $this->_template->render(false, false, 'json-success.php');
    }

    public function addRemoveWishListProduct($selprod_id, $wish_list_id, $rowAction = '')
    {
        $selprod_id = FatUtility::int($selprod_id);
        $wish_list_id = FatUtility::int($wish_list_id);
        $rowAction = ('' == $rowAction ? -1 : $rowAction);

        if (1 > $wish_list_id) {
            $wish_list_id = $this->getDefaultWishListId();
        }
        $loggedUserId = $this->user["user_id"];

        if (1 > $selprod_id) {
            $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        $action = $this->updateWishList($selprod_id, $wish_list_id, $rowAction);

        //UserWishList
        $srch = UserWishList::getSearchObject($loggedUserId);
        $srch->joinTable(UserWishList::DB_TBL_LIST_PRODUCTS, 'LEFT OUTER JOIN', 'uwlist_id = uwlp_uwlist_id');
        $srch->addCondition('uwlp_selprod_id', '=', $selprod_id);
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addMultipleFields(array('uwlist_id'));
        $rs = $srch->getResultSet();
        $row = FatApp::getDb()->fetch($rs);
        $productIsInAnyList = false;
        if ($row) {
            $productIsInAnyList = true;
        }

        $this->set('productIsInAnyList', $productIsInAnyList);
        $this->set('action', $action);
        $this->set('wish_list_id', $wish_list_id);
        $this->set('totalWishListItems', Common::countWishList());
        if (!API_CALL) {
            CommonHelper::deleteCollectionCache();
        }
        $this->_template->render(false, false, 'json-success.php');
    }

    private function updateWishList($selprod_id, $wish_list_id, $rowAction = -1)
    {
        $loggedUserId = $this->user["user_id"];

        $row = false;

        $db = FatApp::getDb();
        $wListObj = new UserWishList();
        if (0 > $rowAction) {
            $srch = UserWishList::getSearchObject($loggedUserId);
            $wListObj->joinWishListProducts($srch);
            $srch->addMultipleFields(array('uwlist_id'));
            $srch->doNotCalculateRecords();
            $srch->doNotLimitRecords();
            $srch->addCondition('uwlp_selprod_id', '=', $selprod_id);
            $srch->addCondition('uwlp_uwlist_id', '=', $wish_list_id);

            $rs = $srch->getResultSet();
            $row = $db->fetch($rs);
        }

        $action = 'N'; //nothing happened
        if (!$row && (0 < $rowAction || 0 > $rowAction)) {
            if (!$wListObj->addUpdateListProducts($wish_list_id, $selprod_id)) {
                $message = Labels::getLabel('LBL_Some_problem_occurred,_Please_contact_webmaster', $this->siteLangId);
                if (API_CALL) {
                    $this->renderJsonError($message);
                }
                Message::addErrorMessage($message);
                FatUtility::dieWithError(Message::getHtml());
            }
            $action = 'A'; //Added to wishlist
            $this->set('msg', Labels::getLabel('LBL_Product_Added_in_list_successfully', $this->siteLangId));
        } else {
            $uwlistIds = array();
            if (API_CALL) {
                $srch = UserWishList::getSearchObject($loggedUserId);
                $srch->addMultipleFields(array('uwlist_id'));
                $rs = $srch->getResultSet();
                $row = $db->fetchAll($rs, 'uwlist_id');
                $uwlistIds = array_keys($row);
            } else {
                $uwlistIds[] = $wish_list_id;
            }
            $err = true;
            foreach ($uwlistIds as $uwlistId) {
                $err = false;
                if (!$db->deleteRecords(UserWishList::DB_TBL_LIST_PRODUCTS, array('smt' => 'uwlp_uwlist_id = ? AND uwlp_selprod_id = ?', 'vals' => array($uwlistId, $selprod_id)))) {
                    $err = true;
                    break;
                }
            }

            if (true == $err) {
                $message = Labels::getLabel('LBL_Some_problem_occurred,_Please_contact_webmaster', $this->siteLangId);
                if (API_CALL) {
                    $this->renderJsonError($message);
                }
                Message::addErrorMessage($message);
                FatUtility::dieWithError(Message::getHtml());
            }

            $action = 'R'; //Removed from wishlist
            $this->set('msg', Labels::getLabel('LBL_Product_Removed_from_list_successfully', $this->siteLangId));
        }
        return $action;
    }

    public function wishlist()
    {
        $this->_template->addJs('js/slick.js');
        $this->_template->render();
    }

    public function wishListSearch()
    {
        $loggedUserId = $this->user["user_id"];

        if (FatApp::getConfig('CONF_ADD_FAVORITES_TO_WISHLIST', FatUtility::VAR_INT, 1) == AppConstants::NO) {
            $wishLists[] = Product::getUserFavouriteProducts($loggedUserId, $this->siteLangId);
        } else {
            $wishLists = UserWishList::getUserWishLists($loggedUserId, false);
            if ($wishLists && false === API_CALL) {
                $srchObj = new UserWishListProductSearch($this->siteLangId);
                $db = FatApp::getDb();
                foreach ($wishLists as &$wishlist) {
                    $srch = clone $srchObj;
                    $srch->joinSellerProducts();
                    $srch->joinProducts();
                    $srch->joinBrands();
                    $srch->joinSellers();
                    $srch->joinShops();
                    $srch->joinProductToCategory();
                    $srch->joinSellerSubscription($this->siteLangId, true);
                    $srch->addSubscriptionValidCondition();
                    $srch->joinSellerProductSpecialPrice();
                    $srch->joinFavouriteProducts($loggedUserId);
                    $srch->addCondition('uwlp_uwlist_id', '=', $wishlist['uwlist_id']);
                    $srch->addCondition('selprod_deleted', '=', AppConstants::NO);
                    $srch->addCondition('selprod_active', '=', AppConstants::YES);
                    $srch->setPageNumber(1);
                    $srch->setPageSize(4);
                    $srch->addMultipleFields(
                        array(
                            'shop_is_buy_now', 'shop_hide_price_for_guests', 'shop_hide_price_for_loginusers', 'selprod_updated_on', 'selprod_type', 'selprod_id', 'IFNULL(selprod_title ,IFNULL(product_name, product_identifier)) as selprod_title',
                            'product_id', 'IFNULL(product_name, product_identifier) as product_name',
                            'IF(selprod_stock > 0, 1, 0) AS in_stock', 'selprod_type'
                        )
                    );
                    $srch->addOrder('uwlp_added_on');
                    $srch->addGroupBy('selprod_id');
                    $rs = $srch->getResultSet();
                    $products = $db->fetchAll($rs);
                    $wishlist['products'] = $products;
                    $wishlist['totalProducts'] = $srch->recordCount();
                }
            }
        }
        $this->set('wishLists', $wishLists);
        $frm = $this->getCreateWishListForm();
        $this->set('frm', $frm);
        $this->_template->render(false, false);
    }

    public function viewFavouriteItems()
    {
        $favouriteListRow = Product::getUserFavouriteProducts($this->user["user_id"], $this->siteLangId);
        if (!$favouriteListRow) {
            Message::addErrorMessage(Labels::getLabel('LBL_Invalid_Request', $this->siteLangId));
            FatUtility::dieWithError(Message::getHtml());
        }
        $this->set('wishListRow', $favouriteListRow);
        $this->_template->render(false, false, 'account/favourite-list-items.php');
    }

    public function searchWishListItems()
    {
        $post = FatApp::getPostedData();
        $db = FatApp::getDb();
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pageSize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $uwlist_id = empty($post['uwlist_id']) ? 0 : FatUtility::int($post['uwlist_id']);
        if (false === API_CALL) {
            $wishListRow = UserWishList::getAttributesById($uwlist_id, array('uwlist_id'));
            if (!$wishListRow) {
                $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
                Message::addErrorMessage($message);
                FatUtility::dieWithError(Message::getHtml());
            }
        }

        $srch = new UserWishListProductSearch($this->siteLangId);
        $srch->joinSellerProducts();
        $srch->joinProducts();
        $srch->joinBrands();
        $srch->joinSellers();
        $srch->joinShops();
        $srch->joinSellerReturnAddress();
        $srch->joinProductToCategory();
        $srch->joinSellerSubscription($this->siteLangId, true);
        $srch->addSubscriptionValidCondition();
        $srch->joinSellerProductSpecialPrice();
        $srch->joinFavouriteProducts($this->user["user_id"]);
        if (API_CALL && 0 >= $uwlist_id) {
            $srch->joinWishLists();
            $srch->addCondition('uwlist_user_id', '=', $this->user["user_id"]);
        } else {
            $srch->addCondition('uwlp_uwlist_id', '=', $uwlist_id);
        }
        $srch->addCondition('selprod_deleted', '=', AppConstants::NO);
        $srch->addCondition('selprod_active', '=', AppConstants::YES);
        $selProdReviewObj = new SelProdReviewSearch();
        $selProdReviewObj->joinSellerProducts();
        $selProdReviewObj->joinSelProdRating();
        $selProdReviewObj->addCondition('sprating_rating_type', '=', SelProdRating::TYPE_PRODUCT);
        $selProdReviewObj->doNotCalculateRecords();
        $selProdReviewObj->doNotLimitRecords();
        $selProdReviewObj->addGroupBy('spr.spreview_product_id');
        $selProdReviewObj->addCondition('spr.spreview_status', '=', SelProdReview::STATUS_APPROVED);
        $selProdReviewObj->addMultipleFields(array('spr.spreview_selprod_id', "ROUND(AVG(sprating_rating),2) as prod_rating"));
        $selProdRviewSubQuery = $selProdReviewObj->getQuery();
        $srch->joinTable('(' . $selProdRviewSubQuery . ')', 'LEFT OUTER JOIN', 'sq_sprating.spreview_selprod_id = selprod_id', 'sq_sprating');
        $srch->setPageNumber($page);
        $srch->setPageSize($pageSize);
        /* groupby added, beacouse if same product is linked with multiple categories, then showing in repeat for each category[ */
        $srch->addGroupBy('selprod_id');
        /* ] */
        $srch->addMultipleFields(
            array(
                'shop_is_buy_now', 'shop_hide_price_for_guests', 'shop_hide_price_for_loginusers', 'selprod_updated_on', 'selprod_type', 'selprod_id', 'IFNULL(selprod_title  ,IFNULL(product_name, product_identifier)) as selprod_title',
                'product_id', 'prodcat_id', 'ufp_id', 'IFNULL(product_name, product_identifier) as product_name', 'IFNULL(prodcat_name, prodcat_identifier) as prodcat_name', 'product_updated_on',
                'IF(selprod_stock > 0, 1, 0) AS in_stock', 'brand.brand_id', 'product_model',
                'IFNULL(brand_name, brand_identifier) as brand_name', 'IFNULL(splprice_price, selprod_price) AS theprice', 'splprice_display_list_price', 'splprice_display_dis_val', 'splprice_display_dis_type',
                'CASE WHEN splprice_selprod_id IS NULL THEN 0 ELSE 1 END AS special_price_found', 'selprod_price',
                'selprod_user_id', 'selprod_code', 'selprod_sold_count', 'selprod_condition',
                'IFNULL(uwlp.uwlp_selprod_id, 0) as is_in_any_wishlist',
                'IFNULL(uwlp.uwlp_uwlist_id, 0) as uwlp_uwlist_id', 'ifnull(prod_rating,0) prod_rating',
                'selprod_min_order_qty', 'selprod_available_from', 'seller_user.user_is_founding_member as is_founding_member', 'seller_user.user_name',
                'seller_user.user_is_verified as seller_verified', 'rcountry.country_code as seller_country', 'shop_is_buy_now as is_buy_now', 'selprod_user_id', 'shop_id', 'IFNULL(shop_name, shop_identifier) as shop_name', 'selprod_type'
            )
        );
        $srch->addOrder('uwlp_added_on', 'DESC');
        $rs = $srch->getResultSet();
        $products = $db->fetchAll($rs);
        if (count($products)) {
            foreach ($products as &$arr) {
                $arr['options'] = SellerProduct::getSellerProductOptions($arr['selprod_id'], true, $this->siteLangId);
            }
        }

        $startRecord = ($page - 1) * $pageSize + 1;
        $endRecord = $page * $pageSize;
        $totalRecords = $srch->recordCount();
        if ($totalRecords < $endRecord) {
            $endRecord = $totalRecords;
        }

        $this->set('products', $products);
        $this->set('showProductShortDescription', false);
        $this->set('showProductReturnPolicy', false);
        $this->set('colMdVal', 6);
        $this->set('page', $page);
        $this->set('recordCount', $srch->recordCount());
        $this->set('pageCount', $srch->pages());
        $this->set('postedData', $post);
        $this->set('totalRecords', $totalRecords);
        $this->set('startRecord', $startRecord);
        $this->set('endRecord', $endRecord);
        $this->set('showActionBtns', true);
        $this->set('isWishList', true);
        if ($totalRecords > 0) {
            $this->set('html', $this->_template->render(false, false, 'products/products-list.php', true, false));
        } else {
            $this->set('html', $this->_template->render(false, false, '_partial/no-record-found.php', true, false));
        }
        $this->set('loadMoreBtnHtml', $this->_template->render(false, false, 'products/products-list-load-more-btn.php', true, false));
        $this->_template->render(false, false, 'json-success.php', true, false);
    }

    public function searchFavouriteListItems()
    {
        $post = FatApp::getPostedData();
        $db = FatApp::getDb();
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pageSize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $wishListRow = Product::getUserFavouriteProducts($this->user["user_id"], $this->siteLangId);
        if (!$wishListRow) {
            $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }
        $srch = new UserFavoriteProductSearch($this->siteLangId);
        $srch->setDefinedCriteria($this->siteLangId);
        $srch->joinBrands();
        $srch->joinSellers();
        $srch->joinShops();
        $srch->joinSellerReturnAddress();
        $srch->joinProductToCategory();
        $srch->joinSellerProductSpecialPrice();
        $srch->joinSellerSubscription($this->siteLangId, true);
        $srch->addSubscriptionValidCondition();
        $srch->addCondition('selprod_deleted', '=', AppConstants::NO);
        $wislistPSrchObj = new UserWishListProductSearch();
        $wislistPSrchObj->joinWishLists();
        $wislistPSrchObj->doNotCalculateRecords();
        $wislistPSrchObj->addCondition('uwlist_user_id', '=', $this->user["user_id"]);
        $wishListSubQuery = $wislistPSrchObj->getQuery();
        $srch->joinTable('(' . $wishListSubQuery . ')', 'LEFT OUTER JOIN', 'uwlp.uwlp_selprod_id = selprod_id', 'uwlp');


        $selProdReviewObj = new SelProdReviewSearch();
        $selProdReviewObj->joinSellerProducts();
        $selProdReviewObj->joinSelProdRating();
        $selProdReviewObj->addCondition('sprating_rating_type', '=', SelProdRating::TYPE_PRODUCT);
        $selProdReviewObj->doNotCalculateRecords();
        $selProdReviewObj->doNotLimitRecords();
        $selProdReviewObj->addGroupBy('spr.spreview_product_id');
        $selProdReviewObj->addCondition('spr.spreview_status', '=', SelProdReview::STATUS_APPROVED);
        $selProdReviewObj->addMultipleFields(array('spr.spreview_selprod_id', "ROUND(AVG(sprating_rating),2) as prod_rating"));
        $selProdRviewSubQuery = $selProdReviewObj->getQuery();
        $srch->joinTable('(' . $selProdRviewSubQuery . ')', 'LEFT OUTER JOIN', 'sq_sprating.spreview_selprod_id = selprod_id', 'sq_sprating');
        $srch->setPageNumber($page);
        $srch->setPageSize($pageSize);
        /* groupby added, beacouse if same product is linked with multiple categories, then showing in repeat for each category[ */
        $srch->addGroupBy('selprod_id');
        /* ] */

        $srch->addMultipleFields(
            array(
                'shop_is_buy_now', 'shop_hide_price_for_guests', 'shop_hide_price_for_loginusers', 'selprod_updated_on', 'selprod_id', 'IFNULL(selprod_title  ,IFNULL(product_name, product_identifier)) as selprod_title',
                'product_id', 'prodcat_id', 'ufp_id', 'IFNULL(product_name, product_identifier) as product_name', 'IFNULL(prodcat_name, prodcat_identifier) as prodcat_name', 'product_updated_on',
                'IF(selprod_stock > 0, 1, 0) AS in_stock', 'brand.brand_id', 'product_model',
                'IFNULL(brand_name, brand_identifier) as brand_name', 'IFNULL(splprice_price, selprod_price) AS theprice', 'splprice_display_list_price', 'splprice_display_dis_val', 'splprice_display_dis_type',
                'CASE WHEN splprice_selprod_id IS NULL THEN 0 ELSE 1 END AS special_price_found', 'selprod_price', 'selprod_user_id', 'selprod_code', 'selprod_condition', 'IFNULL(uwlp.uwlp_selprod_id, 0) as is_in_any_wishlist', 'ifnull(prod_rating,0) prod_rating', 'selprod_sold_count', 'selprod_min_order_qty', 'selprod_available_from', 'seller_user.user_is_founding_member as is_founding_member', 'seller_user.user_name', 'seller_user.user_is_verified as seller_verified', 'rcountry.country_code as seller_country', 'shop_is_buy_now as is_buy_now', 'shop_id', 'IFNULL(shop_name, shop_identifier) as shop_name', 'selprod_type'
            )
        );

        $srch->addOrder('ufp_id', 'desc');
        $srch->addCondition('ufp_user_id', '=', $this->user["user_id"]);
        $rs = $srch->getResultSet();
        $products = $db->fetchAll($rs);
        $this->set('products', $products);
        $this->set('showProductShortDescription', false);
        $this->set('showProductReturnPolicy', false);
        $this->set('colMdVal', 2);
        $this->set('page', $page);
        $this->set('pagingFunc', 'goToFavouriteListingSearchPage');
        $this->set('recordCount', $srch->recordCount());
        $this->set('pageCount', $srch->pages());
        $this->set('postedData', $post);
        $this->set('showActionBtns', true);
        $startRecord = ($page - 1) * $pageSize + 1;
        $endRecord = $page * $pageSize;
        $totalRecords = $srch->recordCount();
        if ($totalRecords < $endRecord) {
            $endRecord = $totalRecords;
        }

        $this->set('totalRecords', $totalRecords);
        $this->set('startRecord', $startRecord);
        $this->set('endRecord', $endRecord);
        if ($totalRecords > 0) {
            $this->set('html', $this->_template->render(false, false, 'products/products-list.php', true, false));
        } else {
            $this->set('html', $this->_template->render(false, false, '_partial/no-record-found.php', true, false));
        }
        $this->set('loadMoreBtnHtml', $this->_template->render(false, false, 'products/products-list-load-more-btn.php', true, false));
        $this->_template->render(false, false, 'json-success.php', true, false);
    }

    public function deleteWishList()
    {
        $uwlist_id = FatApp::getPostedData('uwlist_id', FatUtility::VAR_INT, 0);
        if (0 >= $uwlist_id) {
            $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        $srch = UserWishList::getSearchObject($this->user["user_id"]);
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addCondition('uwlist_id', '=', $uwlist_id);
        $srch->addCondition('uwlist_default', '=', AppConstants::NO);
        $rs = $srch->getResultSet();
        $row = FatApp::getDb()->fetch($rs);
        if (!$row) {
            $message = Labels::getLabel('MSG_No_record_found', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        $obj = new UserWishList();
        $obj->deleteWishList($row['uwlist_id']);
        $this->renderJsonSuccess(Labels::getLabel('LBL_Record_deleted_successfully', $this->siteLangId));
    }

    public function viewWishListItems()
    {
        $post = FatApp::getPostedData();
        $uwlist_id = FatUtility::int($post['uwlist_id']);
        $db = FatApp::getDb();
        $srch = UserWishList::getSearchObject($this->user["user_id"]);
        $srch->addMultipleFields(array('uwlist_id', 'uwlist_title', 'uwlist_default'));
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addCondition('uwlist_id', '=', $uwlist_id);
        $rs = $srch->getResultSet();
        $wishListRow = $db->fetch($rs);
        if (!$wishListRow) {
            Message::addErrorMessage(Labels::getLabel('LBL_Invalid_Request', $this->siteLangId));
            FatUtility::dieWithError(Message::getHtml());
        }
        $this->set('wishListRow', $wishListRow);
        $this->_template->render(false, false, 'account/wish-list-items.php');
    }

    public function toggleShopFavorite()
    {
        $shop_id = FatApp::getPostedData('shop_id', FatUtility::VAR_INT, 0);
        $db = FatApp::getDb();
        $srch = new ShopSearch($this->siteLangId);
        $srch->setDefinedCriteria($this->siteLangId);
        $srch->joinSellerSubscription();
        $srch->doNotCalculateRecords();
        $srch->addMultipleFields(
            array(
                'shop_id', 'shop_user_id', 'shop_ltemplate_id', 'shop_created_on', 'shop_name', 'shop_description',
                'shop_country_l.country_name as shop_country_name', 'shop_state_l.state_name as shop_state_name', 'shop_city'
            )
        );
        $srch->addCondition('shop_id', '=', $shop_id);
        $shopRs = $srch->getResultSet();
        $shop = $db->fetch($shopRs);
        if (!$shop) {
            $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        $action = 'N'; //nothing happened
        $srch = new UserFavoriteShopSearch();
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addCondition('ufs_user_id', '=', $this->user["user_id"]);
        $srch->addCondition('ufs_shop_id', '=', $shop_id);
        $rs = $srch->getResultSet();
        if (!$row = $db->fetch($rs)) {
            $shopObj = new Shop();
            if (!$shopObj->addUpdateUserFavoriteShop($this->user["user_id"], $shop_id)) {
                $message = Labels::getLabel('LBL_Some_problem_occurred,_Please_contact_webmaster', $this->siteLangId);
                if (API_CALL) {
                    $this->renderJsonError($message);
                }
                Message::addErrorMessage($message);
                FatUtility::dieWithError(Message::getHtml());
            }
            $action = 'A'; //Added to favorite
            $this->set('msg', Labels::getLabel('LBL_Shop_is_marked_as_favoutite', $this->siteLangId));
        } else {
            if (!$db->deleteRecords(Shop::DB_TBL_SHOP_FAVORITE, array('smt' => 'ufs_user_id = ? AND ufs_shop_id = ?', 'vals' => array($this->user["user_id"], $shop_id)))) {
                $message = Labels::getLabel('LBL_Some_problem_occurred,_Please_contact_webmaster', $this->siteLangId);
                if (API_CALL) {
                    $this->renderJsonError($message);
                }
                Message::addErrorMessage($message);
                FatUtility::dieWithError(Message::getHtml());
            }
            $action = 'R'; //Removed from favorite
            $this->set('msg', Labels::getLabel('LBL_Shop_has_been_removed_from_your_favourite_list', $this->siteLangId));
        }

        $this->set('action', $action);
        $this->_template->render(false, false, 'json-success.php');
    }

    public function favoriteShopSearch()
    {
        $post = FatApp::getPostedData();
        $page = max(1, FatApp::getPostedData('page', FatUtility::VAR_INT, 1));
        $pageSize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $db = FatApp::getDb();
        $srch = new UserFavoriteShopSearch($this->siteLangId);
        $srch->setDefinedCriteria();
        $srch->joinSellerOrder();
        $srch->joinSellerOrderSubscription($this->siteLangId);
        $srch->addCondition('ufs_user_id', '=', $this->user["user_id"]);
        $srch->addMultipleFields(
            array(
                's.shop_id', 'shop_user_id', 'shop_ltemplate_id', 'shop_created_on', 'shop_name', 'shop_description',
                'shop_country_l.country_name as country_name', 'shop_state_l.state_name as state_name', 'shop_city',
                'IFNULL(ufs.ufs_id, 0) as is_favorite'
            )
        );
        $srch->setPageNumber($page);
        $srch->setPageSize($pageSize);
        $rs = $srch->getResultSet();
        $shops = $db->fetchAll($rs);
        if ($shops) {
            foreach ($shops as &$shop) {
                $shop['shopRating'] = SelProdRating::getSellerRating($shop['shop_user_id']);
            }
        }
        $this->set('page', $page);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('postedData', $post);
        $this->set('shops', $shops);
        $this->_template->render(false, false);
    }

    private function isValidSelProd($selprodId)
    {
        $db = FatApp::getDb();
        $srch = new ProductSearch($this->siteLangId);
        $srch->setDefinedCriteria(0, 0, array(), false);
        $srch->doNotCalculateRecords();
        $srch->addMultipleFields(['selprod_id']);
        $srch->addCondition('selprod_id', '=', $selprodId);
        $srch->joinProductToCategory();
        $srch->joinShops();
        $srch->joinSellerSubscription();
        $srch->addSubscriptionValidCondition();
        $srch->addCondition('selprod_deleted', '=', AppConstants::NO);
        $productRs = $srch->getResultSet();
        $product = $db->fetch($productRs);
        if (!$product) {
            $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }
        return true;
    }

    public function toggleProductStatus(int $selprodId, int $status)
    {
        $this->isValidSelProd($selprodId);
        switch ($status) {
            case AppConstants::ACTIVE:
                $this->markAsFavorite($selprodId, false);
                $this->set('msg', Labels::getLabel('MSG_Product_has_been_marked_as_favourite_successfully', $this->siteLangId));
                break;
            case AppConstants::INACTIVE:
                $this->removeFromFavorite($selprodId, false);
                $this->set('msg', Labels::getLabel('MSG_Product_has_been_removed_from_favourite_list', $this->siteLangId));
                break;
            default:
                $this->renderJsonError(Labels::getLabel('MSG_UNKNOWN_ACTION', $this->siteLangId));
                break;
        }
        $this->_template->render();
    }

    public function markAsFavorite($selprodId, $renderView = true)
    {
        $this->isValidSelProd($selprodId);
        $prodObj = new Product();
        if (!$prodObj->addUpdateUserFavoriteProduct($this->user["user_id"], $selprodId)) {
            $message = Labels::getLabel('LBL_Some_problem_occurred,_Please_contact_webmaster', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        if (false === $renderView) {
            return true;
        }
        $this->renderJsonSuccess(Labels::getLabel('LBL_Product_has_been_marked_as_favourite_successfully', $this->siteLangId));
    }

    public function removeFromFavorite($selprodId, $renderView = true)
    {
        $this->isValidSelProd($selprodId);
        $db = FatApp::getDb();
        $loggedUserId = $this->user["user_id"];
        if (!$db->deleteRecords(Product::DB_TBL_PRODUCT_FAVORITE, array('smt' => 'ufp_user_id = ? AND ufp_selprod_id = ?', 'vals' => array($loggedUserId, $selprodId)))) {
            $message = Labels::getLabel('LBL_Some_problem_occurred,_Please_contact_webmaster', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        if (false === $renderView) {
            return true;
        }
        $this->renderJsonSuccess(Labels::getLabel('LBL_Product_has_been_removed_from_favourite_list', $this->siteLangId));
    }

    public function removeFromFavoriteArr()
    {
        $selprodIdsArr = (array) FatApp::getPostedData('selprod_id', FatUtility::VAR_INT);
        if (empty($selprodIdsArr)) {
            $message = Labels::getLabel('LBL_Invalid_Request', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        foreach ($selprodIdsArr as $selprodId) {
            $this->removeFromFavorite($selprodId, false);
        }
        $this->renderJsonSuccess(Labels::getLabel('LBL_Product_has_been_removed_from_favourite_list', $this->siteLangId));
        
    }

    public function messages()
    {
        $this->userPrivilege->canViewMessages($this->user["user_id"]);
        $frm = $this->getMessageSearchForm($this->siteLangId);
        $this->set('frmSrch', $frm);
        $this->_template->render();
    }

    public function messageSearch()
    {
        $userImgUpdatedOn = User::getAttributesById($this->user["user_id"], 'user_updated_on');
        $uploadedTime = AttachedFile::setTimeParam($userImgUpdatedOn);
        $frm = $this->getMessageSearchForm($this->siteLangId);
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());

        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pagesize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $parentAndTheirChildIds = User::getParentAndTheirChildIds($this->userParentId, false, true);
        
        $srch = new MessageSearch();
        $srch->joinThreadLastMessage();
        $srch->joinMessagePostedFromUser(true, $this->siteLangId);
        $srch->joinMessagePostedToUser(true, $this->siteLangId);
        $srch->joinThreadStartedByUser();
        $srch->addMultipleFields(array(
            'tth.*',
            'ttm.message_id', 'ttm.message_text', 'ttm.message_date', 'ttm.message_is_unread',
            'ttm.message_to', 'IFNULL(tfrs_l.shop_name, tfrs.shop_identifier) as message_from_shop_name',
            'tfrs.shop_id as message_from_shop_id', 'tftos.shop_id as message_to_shop_id',
            'IFNULL(tftos_l.shop_name, tftos.shop_identifier) as message_to_shop_name'
        ));
        $srch->addCondition('ttm.message_deleted', '=', 0);
        $cnd = $srch->addCondition('ttm.message_from', 'IN', $parentAndTheirChildIds);
        $cnd->attachCondition('ttm.message_to', 'IN', $parentAndTheirChildIds, 'OR');
        $srch->addOrder('message_id', 'DESC');
        $srch->addGroupBy('ttm.message_thread_id');
        if ($post['keyword'] != '') {
            $cnd = $srch->addCondition('tth.thread_subject', 'like', "%" . $post['keyword'] . "%");
            $cnd->attachCondition('tfr.user_name', 'like', "%" . $post['keyword'] . "%", 'OR');
            $cnd->attachCondition('tfr_c.credential_username', 'like', "%" . $post['keyword'] . "%", 'OR');
        }
        if (!empty($post['thread_type'])) {
            if ($post['thread_type'] == 1) {
                $srch->addCondition('tth.thread_type', '!=', Thread::THREAD_TYPE_RFQ);
            } elseif ($post['thread_type'] == 2) {
                $srch->addCondition('tth.thread_type', '=', Thread::THREAD_TYPE_RFQ);
            }
        }

        $page = (empty($page) || $page <= 0) ? 1 : $page;
        $page = FatUtility::int($page);
        $srch->setPageNumber($page);
        $srch->setPageSize($pagesize);
        $rs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($rs);

        if (API_CALL) {
            $message_records = array();
            foreach ($records as $mkey => $mval) {
                $profile_images_arr = array(
                    "message_from_profile_url" => FatCache::getCachedUrl(CommonHelper::generateFullFileUrl('image', 'user', array($mval['message_from_user_id'], 'thumb', 1)) . $uploadedTime, CONF_IMG_CACHE_TIME, '.jpg'),
                    "message_to_profile_url" => FatCache::getCachedUrl(CommonHelper::generateFullFileUrl('image', 'user', array($mval['message_to_user_id'], 'thumb', 1)) . $uploadedTime, CONF_IMG_CACHE_TIME, '.jpg'),
                    "message_timestamp" => strtotime($mval['message_date'])
                );
                $message_records[] = array_merge($mval, $profile_images_arr);
            }
            $records = $message_records;
        }

        /* CommonHelper::printArray($records); die; */
        $this->set("arr_listing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('loggedUserId', $this->user["user_id"]);
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('parentAndTheirChildIds', $parentAndTheirChildIds);
        $this->set('postedData', $post);

        if (API_CALL) {
            $this->_template->render();
        }
        $this->_template->render(false, false);
    }

    public function viewMessages($threadId, $messageId = 0)
    {
        $this->userPrivilege->canViewMessages($this->user["user_id"]);
        $threadId = FatUtility::int($threadId);
        $messageId = FatUtility::int($messageId);
        if (1 > $threadId) {
            $message = Labels::getLabel('MSG_INVALID_ACCESS', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            CommonHelper::redirectUserReferer();
        }

        $threadData = Thread::getAttributesById($messageId, array('thread_id,thread_type'));
        if ($threadData == false) {
            $message = Labels::getLabel('MSG_INVALID_ACCESS', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            CommonHelper::redirectUserReferer();
        }

        $srch = new MessageSearch();

        $srch->joinThreadMessage();
        $srch->joinMessagePostedFromUser();
        $srch->joinMessagePostedToUser();
        $srch->joinThreadStartedByUser();
        if ($threadData['thread_type'] == Thread::THREAD_TYPE_SHOP) {
            $srch->joinShops($this->siteLangId);
        } elseif ($threadData['thread_type'] == Thread::THREAD_TYPE_PRODUCT) {
            $srch->joinProducts($this->siteLangId);
        } elseif ($threadData['thread_type'] == Thread::THREAD_TYPE_RFQ) {
            $srch->joinRfqProducts($this->siteLangId);
        }

        $parentAndThierChildIds = User::getParentAndTheirChildIds($this->userParentId, false, true);

        $srch->joinOrderProducts();
        $srch->joinOrderProductStatus();
        $srch->addMultipleFields(array('tth.*', 'top.op_invoice_number'));
        $srch->addCondition('ttm.message_deleted', '=', 0);
        $srch->addCondition('tth.thread_id', '=', $threadId);
        if ($messageId) {
            $srch->addCondition('ttm.message_id', '=', $messageId);
        }

        $cnd = $srch->addCondition('ttm.message_from', 'IN', $parentAndThierChildIds);
        $cnd->attachCondition('ttm.message_to', 'IN', $parentAndThierChildIds, 'OR');
        $rs = $srch->getResultSet();
        $threadDetails = FatApp::getDb()->fetch($rs);

        if ($threadDetails == false) {
            $message = Labels::getLabel('MSG_INVALID_ACCESS', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            CommonHelper::redirectUserReferer();
        }

        if (false === API_CALL) {
            $frmSrch = $this->getMsgSearchForm($this->siteLangId);
            $frmSrch->fill(array('thread_id' => $threadId));
            $frm = $this->sendMessageForm($this->siteLangId);
            $frm->fill(array('message_thread_id' => $threadId, 'message_id' => $messageId));
        }

        $threadObj = new Thread($threadId);
        if (!$threadObj->markMessageReadFromUserArr($threadId, $parentAndThierChildIds)) {
            if (API_CALL) {
                $this->renderJsonError($threadObj->getError());
            }
            Message::addErrorMessage($threadObj->getError());
            CommonHelper::redirectUserReferer();
        }

        $this->set('frmSrch', $frmSrch);
        $this->set('frm', $frm);
        $this->set('canEditMessages', $this->userPrivilege->canEditMessages($this->user["user_id"], true));
        $this->set('threadDetails', $threadDetails);
        $this->set('threadTypeArr', Thread::getThreadTypeArr($this->siteLangId));
        $this->set('loggedUserId', $this->user["user_id"]);
        $this->set('loggedUserName', ucfirst(UserAuthentication::getLoggedUserAttribute('user_name')));
        $this->_template->render();
    }

    public function threadMessageSearch()
    {
        $this->userPrivilege->canViewMessages($this->user["user_id"]);
        $post = FatApp::getPostedData();
        $threadId = FatApp::getPostedData('thread_id', FatUtility::VAR_INT, 0);
        if (1 > $threadId) {
            $this->renderJsonError(Labels::getLabel('MSG_INVALID_ACCESS', $this->siteLangId));
        }

        if (API_CALL) {
            $threadObj = new Thread($threadId);
            if (!$threadObj->markUserMessageRead($threadId, $this->user["user_id"])) {
                $this->renderJsonError($threadObj->getError());
            }
        }

        $allowedUserIds = User::getParentAndTheirChildIds($this->userParentId, false, true);
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pagesize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);

        $srch = new MessageSearch();
        $srch->joinThreadMessage();
        $srch->joinMessagePostedFromUser(true, $this->siteLangId);
        $srch->joinMessagePostedToUser(true, $this->siteLangId);
        $srch->joinThreadStartedByUser();
        $srch->addMultipleFields(array(
            'tth.*', 'ttm.message_id', 'ttm.message_text', 'ttm.message_date', 'ttm.message_is_unread',
            'IFNULL(tfrs_l.shop_name, tfrs.shop_identifier) as message_from_shop_name', 'tfrs.shop_id as message_from_shop_id',
            'tftos.shop_id as message_to_shop_id', 'IFNULL(tftos_l.shop_name, tftos.shop_identifier) as message_to_shop_name'
        ));
        $srch->addCondition('ttm.message_deleted', '=', 0);
        $srch->addCondition('tth.thread_id', '=', $threadId);
        $cnd = $srch->addCondition('ttm.message_from', 'in', $allowedUserIds);
        $cnd->attachCondition('ttm.message_to', 'in', $allowedUserIds, 'OR');
        $srch->addOrder('message_id', 'DESC');
        $srch->setPageNumber($page);
        $srch->setPageSize($pagesize);
        $rs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($rs, 'message_id');

        ksort($records);

        $this->set("arrListing", $records);
        $this->set('pageCount', $srch->pages());
        $this->set('recordCount', $srch->recordCount());
        $this->set('page', $page);
        $this->set('pageSize', $pagesize);
        $this->set('postedData', $post);

        $startRecord = ($page - 1) * $pagesize + 1;
        $endRecord = $pagesize;
        $totalRecords = $srch->recordCount();
        if ($totalRecords < $endRecord) {
            $endRecord = $totalRecords;
        }

        $this->set('totalRecords', $totalRecords);
        $this->set('startRecord', $startRecord);
        $this->set('endRecord', $endRecord);
        $this->set('records', $records);

        if (API_CALL) {
            $this->_template->render();
        }

        $this->set('loadMoreBtnHtml', $this->_template->render(false, false, '_partial/load-previous-btn.php', true));
        $this->set('html', $this->_template->render(false, false, 'account/thread-message-search.php', true, false));
        $this->_template->render(false, false, 'json-success.php', true, false);
    }

    public function sendMessage()
    {
        $frm = $this->sendMessageForm($this->siteLangId);
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }

        $threadId = FatUtility::int($post['message_thread_id']);
        $messageId = FatUtility::int($post['message_id']);

        if (1 > $threadId || 1 > $messageId) {
            $message = Labels::getLabel('MSG_Invalid_Access', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }
        $allowedUserIds = User::getParentAndTheirChildIds($this->userParentId, false, true);
        $srch = new MessageSearch();
        $srch->joinThreadMessage();
        $srch->joinMessagePostedFromUser();
        $srch->joinMessagePostedToUser();
        $srch->joinThreadStartedByUser();
        $srch->addMultipleFields(array('tth.*'));
        $srch->addCondition('ttm.message_deleted', '=', 0);
        $srch->addCondition('tth.thread_id', '=', $threadId);
        $srch->addCondition('ttm.message_id', '=', $messageId);
        $cnd = $srch->addCondition('ttm.message_from', 'in', $allowedUserIds);
        $cnd->attachCondition('ttm.message_to', 'in', $allowedUserIds, 'OR');
        $rs = $srch->getResultSet();

        $threadDetails = FatApp::getDb()->fetch($rs);
        if (empty($threadDetails)) {
            $message = Labels::getLabel('MSG_Invalid_Access', $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }
        $messageSendTo = ($threadDetails['message_from_user_id'] == $this->user["user_id"] || $threadDetails['message_from_user_id'] == $this->userParentId) ? $threadDetails['message_to_user_id'] : $threadDetails['message_from_user_id'];
        $data = array(
            'message_thread_id' => $threadId,
            'message_from' => $this->user["user_id"],
            'message_to' => $messageSendTo,
            'message_text' => $post['message_text'],
            'message_date' => date('Y-m-d H:i:s'),
            'message_is_unread' => 1
        );

        $tObj = new Thread();
        if (!$insertId = $tObj->addThreadMessages($data)) {
            $message = Labels::getLabel($tObj->getError(), $this->siteLangId);
            if (API_CALL) {
                $this->renderJsonError($message);
            }
            Message::addErrorMessage($message);
            FatUtility::dieWithError(Message::getHtml());
        }

        if ($insertId) {
            $emailObj = new EmailHandler();
            $emailObj->SendMessageNotification($insertId, $this->siteLangId);
        }

        $this->set('threadId', $threadId);
        $this->set('messageId', $insertId);
        $this->set('msg', Labels::getLabel('MSG_Message_Submitted_Successfully!', $this->siteLangId));
        if (API_CALL) {
            $this->set('messageDetail', $data);
            $this->_template->render();
        }
        $this->_template->render(false, false, 'json-success.php');
    }

    private function getMessageSearchForm($langId)
    {
        $frm = new Form('frmMessageSrch');
        $frm->addTextBox('', 'keyword');
        $frm->addSelectBox('Message Type', 'thread_type', array('1' => Labels::getLabel('LBL_Normal_Messages', $langId), '2' => Labels::getLabel('LBL_RFQ_Messages', $langId)));
        $fldSubmit = $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Search', $langId));
        $fldCancel = $frm->addButton("", "btn_clear", Labels::getLabel("LBL_Clear", $langId), array('onclick' => 'clearSearch();'));
        $frm->addHiddenField('', 'page');
        return $frm;
    }

    private function getWithdrawalForm($langId)
    {

        if (User::isUserFromColombiaCountry($this->user["user_id"])) {
            $walletBalance = User::getUserBalanceInColombianCurrency($this->user["user_id"]);
            $str = CommonHelper::displayMoneyFormat($walletBalance, true, false, true, false, false, Currency::COLOMBIAN_CURRENCY_CODE);
            $currSymbol = commonHelper::getColombianCurrencySymbol();
        } else {
            $walletBalance = User::getUserBalance($this->user["user_id"]);
            $str = CommonHelper::displayMoneyFormat($walletBalance, true, true);
            $currSymbol = commonHelper::getDefaultCurrencySymbol();
        }

        $frm = new Form('frmWithdrawal');
        $fld = $frm->addRequiredField(Labels::getLabel('LBL_Amount_to_be_Withdrawn', $langId) . ' [' . $currSymbol . ']', 'withdrawal_amount');
        $fld->requirement->setFloat(true);


        $fld->htmlAfterField = Labels::getLabel("LBL_Current_Wallet_Balance", $langId) . ' ' . $str;

        if (User::isAffiliate()) {
            $PayMethodFld = $frm->addRadioButtons(Labels::getLabel('LBL_Payment_Method', $langId), 'uextra_payment_method', User::getAffiliatePaymentMethodArr($langId));

            /* [ */
            $frm->addTextBox(Labels::getLabel('LBL_Cheque_Payee_Name', $langId), 'uextra_cheque_payee_name');
            $chequePayeeNameUnReqFld = new FormFieldRequirement('uextra_cheque_payee_name', Labels::getLabel('LBL_Cheque_Payee_Name', $langId));
            $chequePayeeNameUnReqFld->setRequired(false);

            $chequePayeeNameReqFld = new FormFieldRequirement('uextra_cheque_payee_name', Labels::getLabel('LBL_Cheque_Payee_Name', $langId));
            $chequePayeeNameReqFld->setRequired(true);

            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_CHEQUE, 'eq', 'uextra_cheque_payee_name', $chequePayeeNameReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_BANK, 'eq', 'uextra_cheque_payee_name', $chequePayeeNameUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_PAYPAL, 'eq', 'uextra_cheque_payee_name', $chequePayeeNameUnReqFld);
            /* ] */

            /* [ */
            $frm->addTextBox(Labels::getLabel('LBL_Bank_Name', $langId), 'ub_bank_name');
            $bankNameUnReqFld = new FormFieldRequirement('ub_bank_name', Labels::getLabel('LBL_Bank_Name', $langId));
            $bankNameUnReqFld->setRequired(false);

            $bankNameReqFld = new FormFieldRequirement('ub_bank_name', Labels::getLabel('LBL_Bank_Name', $langId));
            $bankNameReqFld->setRequired(true);

            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_CHEQUE, 'eq', 'ub_bank_name', $bankNameUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_BANK, 'eq', 'ub_bank_name', $bankNameReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_PAYPAL, 'eq', 'ub_bank_name', $bankNameUnReqFld);
            /* ] */

            /* [ */
            $frm->addTextBox(Labels::getLabel('LBL_Account_Holder_Name', $langId), 'ub_account_holder_name');
            $bankAccHolderNameUnReqFld = new FormFieldRequirement('ub_account_holder_name', Labels::getLabel('LBL_Account_Holder_Name', $langId));
            $bankAccHolderNameUnReqFld->setRequired(false);

            $bankAccHolderNameReqFld = new FormFieldRequirement('ub_account_holder_name', Labels::getLabel('LBL_Account_Holder_Name', $langId));
            $bankAccHolderNameReqFld->setRequired(true);

            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_CHEQUE, 'eq', 'ub_account_holder_name', $bankAccHolderNameUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_BANK, 'eq', 'ub_account_holder_name', $bankAccHolderNameReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_PAYPAL, 'eq', 'ub_account_holder_name', $bankAccHolderNameUnReqFld);
            /* ] */

            /* [ */
            $frm->addTextBox(Labels::getLabel('LBL_Bank_Account_Number', $langId), 'ub_account_number');
            $bankAccNumberUnReqFld = new FormFieldRequirement('ub_account_number', Labels::getLabel('LBL_Bank_Account_Number', $langId));
            $bankAccNumberUnReqFld->setRequired(false);

            $bankAccNumberReqFld = new FormFieldRequirement('ub_account_number', Labels::getLabel('LBL_Bank_Account_Number', $langId));
            $bankAccNumberReqFld->setRequired(true);

            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_CHEQUE, 'eq', 'ub_account_number', $bankAccNumberUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_BANK, 'eq', 'ub_account_number', $bankAccNumberReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_PAYPAL, 'eq', 'ub_account_number', $bankAccNumberUnReqFld);
            /* ] */

            /* [ */
            $frm->addTextBox(Labels::getLabel('LBL_Swift_Code', $langId), 'ub_ifsc_swift_code');
            $bankIfscUnReqFld = new FormFieldRequirement('ub_ifsc_swift_code', Labels::getLabel('LBL_Swift_Code', $langId));
            $bankIfscUnReqFld->setRequired(false);

            $bankIfscReqFld = new FormFieldRequirement('ub_ifsc_swift_code', Labels::getLabel('LBL_Swift_Code', $langId));
            $bankIfscReqFld->setRequired(true);

            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_CHEQUE, 'eq', 'ub_ifsc_swift_code', $bankIfscUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_BANK, 'eq', 'ub_ifsc_swift_code', $bankIfscReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_PAYPAL, 'eq', 'ub_ifsc_swift_code', $bankIfscUnReqFld);
            /* ] */

            /* [ */
            $frm->addTextArea(Labels::getLabel('LBL_Bank_Address', $langId), 'ub_bank_address');
            $bankBankAddressUnReqFld = new FormFieldRequirement('ub_bank_address', Labels::getLabel('LBL_Bank_Address', $langId));
            $bankBankAddressUnReqFld->setRequired(false);

            $bankBankAddressReqFld = new FormFieldRequirement('ub_bank_address', Labels::getLabel('LBL_Bank_Address', $langId));
            $bankBankAddressReqFld->setRequired(true);

            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_CHEQUE, 'eq', 'ub_bank_address', $bankBankAddressUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_BANK, 'eq', 'ub_bank_address', $bankBankAddressReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_PAYPAL, 'eq', 'ub_bank_address', $bankBankAddressUnReqFld);
            /* ] */

            /* [ */
            $fld = $frm->addTextBox(Labels::getLabel('LBL_PayPal_Email_Account', $langId), 'uextra_paypal_email_id');
            $PPEmailIdUnReqFld = new FormFieldRequirement('uextra_paypal_email_id', Labels::getLabel('LBL_PayPal_Email_Account', $langId));
            $PPEmailIdUnReqFld->setRequired(false);

            $PPEmailIdReqFld = new FormFieldRequirement('uextra_paypal_email_id', Labels::getLabel('LBL_PayPal_Email_Account', $langId));
            $PPEmailIdReqFld->setRequired(true);
            $PPEmailIdReqFld->setEmail();

            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_CHEQUE, 'eq', 'uextra_paypal_email_id', $PPEmailIdUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_BANK, 'eq', 'uextra_paypal_email_id', $PPEmailIdUnReqFld);
            $PayMethodFld->requirements()->addOnChangerequirementUpdate(User::AFFILIATE_PAYMENT_METHOD_PAYPAL, 'eq', 'uextra_paypal_email_id', $PPEmailIdReqFld);
            /* ] */
        } else {
            $frm->addRequiredField(Labels::getLabel('LBL_Bank_Name', $langId), 'ub_bank_name');
            $frm->addRequiredField(Labels::getLabel('LBL_Account_Holder_Name', $langId), 'ub_account_holder_name');
            $frm->addRequiredField(Labels::getLabel('LBL_Account_Number', $langId), 'ub_account_number');
            $ifsc = $frm->addRequiredField(Labels::getLabel('LBL_IFSC_Swift_Code', $langId), 'ub_ifsc_swift_code');
            $ifsc->requirements()->setRegularExpressionToValidate(ValidateElement::USERNAME_REGEX);
            $frm->addTextArea(Labels::getLabel('LBL_Bank_Address', $langId), 'ub_bank_address');
        }
        $frm->addTextArea(Labels::getLabel('LBL_Other_Info_Instructions', $langId), 'withdrawal_comments');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Request', $langId));
        $frm->addButton("", "btn_cancel", Labels::getLabel("LBL_Cancel", $langId));
        return $frm;
    }

    private function getCreateWishListForm()
    {
        $frm = new Form('frmCreateWishList');
        $frm->setRequiredStarWith('NONE');
        $frm->addRequiredField('', 'uwlist_title');
        $frm->addHiddenField('', 'selprod_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Add', $this->siteLangId));
        $frm->setJsErrorDisplay('afterfield');
        return $frm;
    }

    private function getProfileInfoForm()
    {
        $frm = new Form('frmProfileInfo');
        $frm->addTextBox(Labels::getLabel('LBL_Username', $this->siteLangId), 'credential_username', '');
        $frm->addTextBox(Labels::getLabel('LBL_Email', $this->siteLangId), 'credential_email', '');
        $frm->addRequiredField(Labels::getLabel('LBL_Customer_Name', $this->siteLangId), 'user_name');
        $frm->addDateField(Labels::getLabel('LBL_Date_Of_Birth', $this->siteLangId), 'user_dob', '', array('readonly' => 'readonly'));
        $phoneFld = $frm->addTextBox(Labels::getLabel('LBL_Phone', $this->siteLangId), 'user_phone', '', array('class' => 'phone-js ltr-right', 'placeholder' => ValidateElement::PHONE_NO_FORMAT, 'maxlength' => ValidateElement::PHONE_NO_LENGTH));
        $phoneFld->requirements()->setRegularExpressionToValidate(ValidateElement::PHONE_REGEX);
        $phoneFld->requirements()->setCustomErrorMessage(Labels::getLabel('LBL_Please_enter_valid_phone_number_format.', $this->siteLangId));
        // $phoneFld->htmlAfterField='<small class="text--small">'.Labels::getLabel('LBL_e.g.', $this->siteLangId).': '.implode(', ', ValidateElement::PHONE_FORMATS).'</small>';

        if (User::isAffiliate()) {
            $frm->addTextBox(Labels::getLabel('LBL_Company', $this->siteLangId), 'uextra_company_name');
            $frm->addTextBox(Labels::getLabel('LBL_Website', $this->siteLangId), 'uextra_website');
            $frm->addTextBox(Labels::getLabel('LBL_Address_Line1', $this->siteLangId), 'user_address1')->requirements()->setRequired();
            $frm->addTextBox(Labels::getLabel('LBL_Address_Line2', $this->siteLangId), 'user_address2');
        }

        $countryObj = new Countries();
        $countriesArr = $countryObj->getCountriesArr($this->siteLangId);
        $fld = $frm->addSelectBox(Labels::getLabel('LBL_Country', $this->siteLangId), 'user_country_id', $countriesArr, FatApp::getConfig('CONF_COUNTRY', FatUtility::VAR_INT, 0), array(), Labels::getLabel('LBL_Select', $this->siteLangId));
        $fld->requirement->setRequired(true);

        $frm->addSelectBox(Labels::getLabel('LBL_State', $this->siteLangId), 'user_state_id', array(), '', array(), Labels::getLabel('LBL_Select', $this->siteLangId))->requirement->setRequired(true);
        $frm->addTextBox(Labels::getLabel('LBL_City', $this->siteLangId), 'user_city');

        if (User::isAffiliate()) {
            $zipFld = $frm->addRequiredField(Labels::getLabel('LBL_Postalcode', $this->siteLangId), 'user_zip');
            $zipFld->requirements()->setRegularExpressionToValidate(ValidateElement::ZIP_REGEX);
            $zipFld->requirements()->setCustomErrorMessage(Labels::getLabel('LBL_Only_alphanumeric_value_is_allowed.', $this->siteLangId));
        }
        $parent = User::getAttributesById(UserAuthentication::getLoggedUserId(true), 'user_parent');
        if (User::isAdvertiser() && $parent == 0) {
            $fld = $frm->addTextBox(Labels::getLabel('L_Company', $this->siteLangId), 'user_company');
            $fld = $frm->addTextArea(Labels::getLabel('L_Brief_Profile', $this->siteLangId), 'user_profile_info');
            $fld->html_after_field = '<small>' . Labels::getLabel('L_Please_tell_us_something_about_yourself', $this->siteLangId) . '</small>';
            $frm->addTextArea(Labels::getLabel('L_What_kind_products_services_advertise', $this->siteLangId), 'user_products_services');
        }

        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_SAVE_CHANGES', $this->siteLangId));
        return $frm;
    }

    private function getProfileImageForm()
    {
        $frm = new Form('frmProfile', array('id' => 'frmProfile'));
        $frm->addFileUpload(Labels::getLabel('LBL_Profile_Picture', $this->siteLangId), 'user_profile_image', array('id' => 'user_profile_image', 'onClick' => 'popupImage(this)', 'accept' => 'image/*', 'data-frm' => 'frmProfile'));
        return $frm;
    }

    private function getBankInfoForm()
    {
        $frm = new Form('frmBankInfo');
        $frm->addRequiredField(Labels::getLabel('M_Bank_Name', $this->siteLangId), 'ub_bank_name', '');
        $frm->addRequiredField(Labels::getLabel('M_Account_Holder_Name', $this->siteLangId), 'ub_account_holder_name', '');
        $fld = $frm->addRequiredField(Labels::getLabel('M_Account_Number', $this->siteLangId), 'ub_account_number', '');
        $fld->requirement->setRequired(true);
        $ifsc = $frm->addRequiredField(Labels::getLabel('M_IFSC_Swift_Code', $this->siteLangId), 'ub_ifsc_swift_code', '');
        $ifsc->requirements()->setRegularExpressionToValidate(ValidateElement::USERNAME_REGEX);
        $frm->addTextArea(Labels::getLabel('M_Bank_Address', $this->siteLangId), 'ub_bank_address', '');
        $frm->addHtml('bank_info_safety_text', 'bank_info_safety_text', '<span class="text--small">' . Labels::getLabel('Lbl_Your_Bank/Card_info_is_safe_with_us', $this->siteLangId) . '</span>');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_SAVE_CHANGES', $this->siteLangId));
        return $frm;
    }

    private function getChangePasswordForm()
    {
        $frm = new Form('changePwdFrm');
        $curPwd = $frm->addPasswordField(Labels::getLabel('LBL_CURRENT_PASSWORD', $this->siteLangId), 'current_password');
        $curPwd->requirements()->setRequired();
        $newPwd = $frm->addPasswordField(Labels::getLabel('LBL_NEW_PASSWORD', $this->siteLangId), 'new_password');
        $newPwd->htmlAfterField = '<span class="text--small">' . sprintf(Labels::getLabel('LBL_Example_password', $this->siteLangId), 'User@123') . '</span>';
        $newPwd->requirements()->setRequired();
        $newPwd->requirements()->setRegularExpressionToValidate(ValidateElement::PASSWORD_REGEX);
        $newPwd->requirements()->setCustomErrorMessage(Labels::getLabel('MSG_PASSWORD_MUST_BE_ATLEAST_EIGHT_CHARACTERS_LONG_AND_ALPHANUMERIC', $this->siteLangId));
        $conNewPwd = $frm->addPasswordField(Labels::getLabel('LBL_CONFIRM_NEW_PASSWORD', $this->siteLangId), 'conf_new_password');
        $conNewPwdReq = $conNewPwd->requirements();
        $conNewPwdReq->setRequired();
        $conNewPwdReq->setCompareWith('new_password', 'eq');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_SAVE', $this->siteLangId));
        return $frm;
    }

    private function notifyAdminSupplierApproval($userObj, $data, $approval_request = 1)
    {
        $attr = array('user_name', 'credential_username', 'credential_email');
        $userData = $userObj->getUserInfo($attr);
        if (empty($userData)) {
            return false;
        }
        $data = array(
            'user_name' => $userData['user_name'],
            'username' => $userData['credential_username'],
            'user_email' => $userData['credential_email'],
            'reference_number' => $data['reference'],
        );
        $email = new EmailHandler();
        if (!$email->sendSupplierApprovalNotification(CommonHelper::getLangId(), $data, $approval_request)) {
            Message::addMessage(Labels::getLabel("MSG_ERROR_IN_SENDING_SUPPLIER_APPROVAL_EMAIL", CommonHelper::getLangId()));
            return false;
        }

        return true;
    }

    private function getSupplierForm()
    {
        $frm = new Form('frmSupplierForm');
        $frm->addHiddenField('', 'id', 0);
        $userObj = new User();
        $supplier_form_fields = $userObj->getSupplierFormFields($this->siteLangId);

        foreach ($supplier_form_fields as $field) {
            $fieldName = 'sformfield_' . $field['sformfield_id'];

            switch ($field['sformfield_type']) {
                case User::USER_FIELD_TYPE_TEXT:
                    $fld = $frm->addTextBox($field['sformfield_caption'], $fieldName);
                    break;

                case User::USER_FIELD_TYPE_TEXTAREA:
                    $fld = $frm->addTextArea($field['sformfield_caption'], $fieldName);
                    break;

                case User::USER_FIELD_TYPE_FILE:
                    $fld1 = $frm->addButton(
                        $field['sformfield_caption'],
                        'button[' . $field['sformfield_id'] . ']',
                        Labels::getLabel('LBL_Upload_File', $this->siteLangId),
                        array('class' => 'fileType-Js', 'id' => 'button-upload' . $field['sformfield_id'], 'data-field_id' => $field['sformfield_id'])
                    );
                    $fld1->htmlAfterField = '<span id="input-sformfield' . $field['sformfield_id'] . '"></span>';
                    if ($field['sformfield_required'] == 1) {
                        $fld1->captionWrapper = array('<div class="astrick">', '</div>');
                    }
                    $fld = $frm->addTextBox('', $fieldName, '', array('id' => $fieldName, 'hidden' => 'hidden', 'title' => $field['sformfield_caption']));
                    $fld->setRequiredStarWith(Form::FORM_REQUIRED_STAR_WITH_NONE);
                    $fld1->attachField($fld);
                    break;

                case User::USER_FIELD_TYPE_DATE:
                    $fld = $frm->addDateField($field['sformfield_caption'], $fieldName, '', array('readonly' => 'readonly'));
                    break;

                case User::USER_FIELD_TYPE_DATETIME:
                    $fld = $frm->addDateTimeField($field['sformfield_caption'], $fieldName, '', array('readonly' => 'readonly'));
                    break;

                case User::USER_FIELD_TYPE_TIME:
                    $fld = $frm->addTextBox($field['sformfield_caption'], $fieldName);
                    $fld->requirements()->setRegularExpressionToValidate(ValidateElement::TIME_REGEX);
                    $fld->htmlAfterField = Labels::getLabel('LBL_HH:MM', $this->siteLangId);
                    $fld->requirements()->setCustomErrorMessage(Labels::getLabel('LBL_Please_enter_valid_time_format.', $this->siteLangId));
                    break;

                case User::USER_FIELD_TYPE_PHONE:
                    $fld = $frm->addTextBox($field['sformfield_caption'], $fieldName, '', array('class' => 'phone-js ltr-right', 'placeholder' => ValidateElement::PHONE_NO_FORMAT, 'maxlength' => ValidateElement::PHONE_NO_LENGTH));
                    $fld->requirements()->setRegularExpressionToValidate(ValidateElement::PHONE_REGEX);
                    break;
            }

            if ($field['sformfield_required'] == 1) {
                $fld->requirements()->setRequired();
            }
            if ($field['sformfield_comment']) {
                $fld->htmlAfterField = '<p class="note">' . $field['sformfield_comment'] . '</p>';
            }
        }
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Save_Changes', $this->siteLangId));
        return $frm;
    }

 
    public function escalateOrderReturnRequest($orrequest_id)
    {
        $orrequest_id = FatUtility::int($orrequest_id);
        if (!$orrequest_id) {
            Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
            CommonHelper::redirectUserReferer();
        }
        
        $srch = new OrderReturnRequestSearch();
        $srch->joinOrderProducts();
        $srch->addCondition('orrequest_id', '=', $orrequest_id);
        $srch->addCondition('orrequest_status', '=', OrderReturnRequest::RETURN_REQUEST_STATUS_PENDING);
        $srch->addCondition('op_selprod_user_id', '=', $this->user['user_id']);
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addMultipleFields(array('orrequest_id', 'orrequest_user_id'));
        $rs = $srch->getResultSet();
        $request = FatApp::getDb()->fetch($rs);
        if (empty($request)) {
            Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
            CommonHelper::redirectUserReferer();
        }

        /* buyer cannot escalate request[ */
        // if( $user_id == $request['orrequest_user_id'] ){
        if (!User::isSeller()) {
            Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Access', $this->siteLangId));
            CommonHelper::redirectUserReferer();
        }
        /* ] */
        
        $orrObj = new OrderReturnRequest();
        if (!$orrObj->escalateRequest($request['orrequest_id'], $this->user["user_id"], $this->siteLangId)) {
            Message::addErrorMessage(Labels::getLabel($orrObj->getError(), $this->siteLangId));
            CommonHelper::redirectUserReferer();
        }

        /* email notification handling[ */
        $emailNotificationObj = new EmailHandler();
        if (!$emailNotificationObj->sendOrderReturnRequestStatusChangeNotification($orrequest_id, $this->siteLangId)) {
            Message::addErrorMessage(Labels::getLabel($emailNotificationObj->getError(), $this->siteLangId));
            CommonHelper::redirectUserReferer();
        }
        /* ] */
        Message::addMessage(Labels::getLabel('MSG_Your_request_sent', $this->siteLangId));
        CommonHelper::redirectUserReferer();
    }

    public function orderReturnRequestMessageSearch()
    {
        $frm = $this->getOrderReturnRequestMessageSearchForm($this->siteLangId);
        $postedData = FatApp::getPostedData();
        $post = $frm->getFormDataFromArray($postedData);
        $page = (empty($post['page']) || $post['page'] <= 0) ? 1 : FatUtility::int($post['page']);
        $pageSize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $user_id = $this->user["user_id"];

        $orrequest_id = isset($post['orrequest_id']) ? FatUtility::int($post['orrequest_id']) : 0;
        $isSeller = isset($postedData['isSeller']) ? FatUtility::int($postedData['isSeller']) : 0;

        $parentAndTheirChildIds = User::getParentAndTheirChildIds($this->userParentId, false, true);
        $srch = new OrderReturnRequestMessageSearch($this->siteLangId);
        $srch->joinOrderReturnRequests();
        $srch->joinMessageUser($this->siteLangId);
        $srch->joinMessageAdmin();
        $srch->joinOrderProducts();
        $srch->addCondition('orrmsg_orrequest_id', '=', $orrequest_id);
        if (0 < $isSeller) {
            $srch->addCondition('op_selprod_user_id', 'in', $parentAndTheirChildIds);
        } else {
            $srch->addCondition('orrequest_user_id', 'in', $parentAndTheirChildIds);
        }
        $srch->setPageNumber($page);
        $srch->setPageSize($pageSize);
        $srch->addOrder('orrmsg_id', 'DESC');
        $srch->addMultipleFields(
            array(
                'orrmsg_id', 'orrmsg_from_user_id', 'orrmsg_msg',
                'orrmsg_date', 'msg_user.user_name as msg_user_name', 'orrequest_status',
                'orrmsg_from_admin_id', 'admin_name', 'ifnull(s_l.shop_name, s.shop_identifier) as shop_name', 's.shop_id', 'op_selprod_user_id'
            )
        );

        $rs = $srch->getResultSet();
        $messagesList = FatApp::getDb()->fetchAll($rs, 'orrmsg_id');
        ksort($messagesList);

        $this->set('messagesList', (!empty($messagesList) ? $messagesList : array()));
        $this->set('page', $page);
        $this->set('pageCount', $srch->pages());
        $this->set('postedData', $post);

        $startRecord = ($page - 1) * $pageSize + 1;
        $endRecord = $page * $pageSize;
        $totalRecords = $srch->recordCount();
        if ($totalRecords < $endRecord) {
            $endRecord = $totalRecords;
        }
        $this->set('totalRecords', $totalRecords);
        $this->set('startRecord', $startRecord);
        $this->set('endRecord', $endRecord);
        $this->set('parentAndTheirChildIds', $parentAndTheirChildIds);

        if (API_CALL) {
            $this->_template->render();
        }

        $this->set('loadMoreBtnHtml', $this->_template->render(false, false, '_partial/load-previous-btn.php', true));
        $this->set('html', $this->_template->render(false, false, 'account/order-return-request-messages-list.php', true, false));
        $this->_template->render(false, false, 'json-success.php', true, false);
    }

    private function getCreditsSearchForm($langId)
    {
        $frm = new Form('frmCreditSrch');
        $frm->addTextBox('', 'keyword', '');
        $frm->addSelectBox('', 'debit_credit_type', array(-1 => Labels::getLabel('LBL_Both-Debit/Credit', $langId)) + Transactions::getCreditDebitTypeArr($langId), -1, array(), '');
        $frm->addDateField('', 'date_from', '', array('readonly' => 'readonly', 'class' => 'field--calender'));
        $frm->addDateField('', 'date_to', '', array('readonly' => 'readonly', 'class' => 'field--calender'));
        /* $frm->addSelectBox( '', 'date_order', array( 'ASC' => Labels::getLabel('LBL_Date_Order_Ascending', $langId), 'DESC' => Labels::getLabel('LBL_Date_Order_Descending', $langId) ), 'DESC', array(), '' ); */
        $fldSubmit = $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Search', $langId));
        $fldCancel = $frm->addButton("", "btn_clear", Labels::getLabel("LBL_Clear", $langId), array('onclick' => 'clearSearch();'));
        $frm->addHiddenField('', 'page');
        return $frm;
    }

    private function sendMessageForm($langId)
    {
        $frm = new Form('frmSendMessage');
        $frm->addTextarea(Labels::getLabel('LBL_Comments', $langId), 'message_text', '')->requirements()->setRequired(true);
        $frm->addHiddenField('', 'message_thread_id');
        $frm->addHiddenField('', 'message_id');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Send', $langId));
        return $frm;
    }

    private function getMsgSearchForm($langId)
    {
        $frm = new Form('frmMessageSrch');
        $frm->addHiddenField('', 'page');
        $frm->addHiddenField('', 'thread_id');
        return $frm;
    }

    private function getSettingsForm()
    {
        $frm = new Form('frmBankInfo');
        $activeInactiveArr = AppConstants::getActiveInactiveArr($this->siteLangId);
        $frm->addSelectBox(Labels::getLabel('LBL_Auto_Renew_Subscription', $this->siteLangId), 'user_autorenew_subscription', $activeInactiveArr, '', array(), Labels::getLabel('LBL_Select', $this->siteLangId));
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_SAVE_CHANGES', $this->siteLangId));
        return $frm;
    }

    private function getRechargeWalletForm($langId)
    {
        $frm = new Form('frmRechargeWallet');
        $fld = $frm->addFloatField('', 'amount');
        //$fld->requirements()->setRequired();
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Add_Credits', $langId));
        return $frm;
    }

    public function myAddresses()
    {
        $this->_template->render();
    }

    public function searchAddresses()
    {
		$callFromRFQpopup = FatApp::getPostedData('callFromRFQpopup');
        $addresses = UserAddress::getUserAddresses($this->user["user_id"], $this->siteLangId);
        if ($addresses) {
            $this->set('addresses', $addresses);
        } else {
            if (API_CALL) {
                $this->set('addresses', array());
            }
            $this->set('noRecordsHtml', $this->_template->render(false, false, '_partial/no-record-found.php', true));
        }
        if (API_CALL) {
            $cartObj = new Cart($this->user["user_id"]);
            $shipping_address_id = $cartObj->getCartShippingAddress();
            $this->set('shippingAddressId', $shipping_address_id);
            $this->_template->render();
        }
		if ($callFromRFQpopup==true) {
           $this->_template->render(false, false,'account/addresses-list-for-popup.php'); 
		   exit;
        }
		
        $this->_template->render(false, false);
    }

    public function addAddressForm($ua_id)
    {
        $ua_id = FatUtility::int($ua_id);
        $addressFrm = $this->getUserAddressForm($this->siteLangId);
        $stateId = 0;
        if ($ua_id > 0) {
            $data = UserAddress::getUserAddresses($this->user["user_id"], $this->siteLangId, 0, $ua_id);
            if ($data === false) {
                $this->renderJsonError(Labels::getLabel('MSG_Invalid_request', $this->siteLangId));
            }
            $stateId = $data['ua_state_id'];
            $addressFrm->fill($data);
        }

        $this->set('ua_id', $ua_id);
        $this->set('stateId', $stateId);
        $this->set('addressFrm', $addressFrm);
        $this->_template->render(false, false);
    }

    public function truncateDataRequestPopup()
    {
        $this->_template->render(false, false);
    }

    public function sendTruncateRequest()
    {
        $srch = new UserGdprRequestSearch();
        $srch->addCondition('ureq_user_id', '=', $this->user['user_id']);
        $srch->addCondition('ureq_type', '=', UserGdprRequest::TYPE_TRUNCATE);
        $srch->addCondition('ureq_status', '=', UserGdprRequest::STATUS_PENDING);
        $srch->addCondition('ureq_deleted', '=', AppConstants::NO);
        $rs = $srch->getResultSet();
        $row = FatApp::getDb()->fetch($rs);
        if ($row) {
            $this->renderJsonError(Labels::getLabel('LBL_You_have_alrady_submitted_the_request', $this->siteLangId));
        }
        $assignValues = array(
            'ureq_user_id' => $this->user['user_id'],
            'ureq_type' => UserGdprRequest::TYPE_TRUNCATE,
            'ureq_date' => date('Y-m-d H:i:s'),
        );

        $userReqObj = new UserGdprRequest();
        $userReqObj->assignValues($assignValues);
        if (!$userReqObj->save()) {
            $this->renderJsonError($userReqObj->getError());
        }
        FatUtility::dieJsonSuccess(Labels::getLabel('MSG_Request_sent_successfully', $this->siteLangId));
    }

    private function getRequestDataForm()
    {
        $frm = new Form('frmRequestdata');
        $frm->addTextBox(Labels::getLabel('LBL_Email', $this->siteLangId), 'credential_email', '', array('readonly' => 'readonly'));
        $frm->addTextBox(Labels::getLabel('LBL_Name', $this->siteLangId), 'user_name', '', array('readonly' => 'readonly'));
        $purposeFld = $frm->addTextArea(Labels::getLabel('LBL_PURPOSE_OF_REQUEST_DATA', $this->siteLangId), 'ureq_purpose');
        $purposeFld->requirements()->setRequired();
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Send_Request', $this->siteLangId));
        return $frm;
    }

    public function requestDataForm()
    {
        $userObj = new User($this->user["user_id"]);
        $srch = $userObj->getUserSearchObj(array('credential_username', 'credential_email', 'user_name'));
        $data = FatApp::getDb()->fetch($srch->getResultSet(), 'user_id');
        if ($data === false) {
            $this->renderJsonError(Labels::getLabel('MSG_INVALID_REQUEST', $this->siteLangId));
        }
        $cPageSrch = ContentPage::getSearchObject($this->siteLangId);
        $cPageSrch->addCondition('cpage_id', '=', FatApp::getConfig('CONF_GDPR_POLICY_PAGE', FatUtility::VAR_INT, 0));
        $cpage = FatApp::getDb()->fetch($cPageSrch->getResultSet());
        $gdprPolicyLinkHref = '';
        if (!empty($cpage) && is_array($cpage)) {
            $gdprPolicyLinkHref = CommonHelper::generateUrl('Cms', 'view', array($cpage['cpage_id']));
        }
        $frm = $this->getRequestDataForm();
        $frm->fill($data);
        $this->set('frm', $frm);
        $this->set('gdprPolicyLinkHref', $gdprPolicyLinkHref);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false);
    }

    public function setupRequestData()
    {
        $frm = $this->getRequestDataForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            $this->renderJsonError(current($frm->getValidationErrors()));
        }
        $srch = new UserGdprRequestSearch();
        $srch->addCondition('ureq_user_id', '=', $this->user["user_id"]);
        $srch->addCondition('ureq_type', '=', UserGdprRequest::TYPE_DATA_REQUEST);
        $srch->addCondition('ureq_status', '=', UserGdprRequest::STATUS_PENDING);
        $srch->addCondition('ureq_deleted', '=', AppConstants::NO);
        $rs = $srch->getResultSet();
        $row = FatApp::getDb()->fetch($rs);
        if ($row) {
            Message::addErrorMessage(Labels::getLabel('LBL_You_have_alrady_submitted_the_data_request', $this->siteLangId));
            FatUtility::dieWithError(Message::getHtml());
        }

        $assignValues = array(
            'ureq_user_id' => $this->user["user_id"],
            'ureq_type' => UserGdprRequest::TYPE_DATA_REQUEST,
            'ureq_date' => date('Y-m-d H:i:s'),
            'ureq_purpose' => $post['ureq_purpose'],
        );

        $userReqObj = new UserGdprRequest();
        $userReqObj->assignValues($assignValues);
        if (!$userReqObj->save()) {
            $this->renderJsonError($userReqObj->getError());
        }
        $post['user_id'] = $this->user["user_id"];
        $emailNotificationObj = new EmailHandler();
        if (!$emailNotificationObj->sendDataRequestNotification($post, $this->siteLangId)) {
            $this->renderJsonError(Labels::getLabel($emailNotificationObj->getError(), $this->siteLangId));
        }

        $this->renderJsonSuccess(Labels::getLabel('MSG_REQUEST_SENT_SUCCESSFULLY', $this->siteLangId));
    }

    //Valid for 10 Minutes only
    public function getTempToken()
    {
        $uObj = new User($this->user["user_id"]);
        $tempToken = substr(md5(rand(1, 99999) . microtime()), 0, UserAuthentication::TOKEN_LENGTH);

        if (!$uObj->createUserTempToken($tempToken)) {
            $this->renderJsonError($uObj->getError());
        }
        $this->set('data', array('tempToken' => $tempToken));
        $this->_template->render();
    }

    public function notifications()
    {
        $page = FatApp::getPostedData('page', FatUtility::VAR_INT, 1);
        $defaultPageSize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $pageSize = FatApp::getPostedData('pagesize', FatUtility::VAR_INT, $defaultPageSize);
        $srch = Notifications::getSearchObject();
        $srch->addCondition('unt.unotification_user_id', '=', $this->user["user_id"]);
        $srch->addOrder('unt.unotification_id', 'DESC');
        $srch->addMultipleFields(array('unt.*'));
        $srch->setPageNumber($page);
        $srch->setPageSize($pageSize);
        $rs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($rs);

        $this->set('notifications', $records);
        $this->set('total_pages', $srch->pages());
        $this->set('total_records', $srch->recordCount());
        $this->_template->render();
    }

    public function readAllNotifications()
    {
        $smt = array(
            'smt' => Notifications::DB_TBL_PREFIX . 'is_read = ? AND ' . Notifications::DB_TBL_PREFIX . 'user_id = ?',
            'vals' => array(AppConstants::NO, (int) $this->user["user_id"])
        );
        $db = FatApp::getDb();
        if (!$db->updateFromArray(Notifications::DB_TBL, array(Notifications::DB_TBL_PREFIX . 'is_read' => 1), $smt)) {
            $this->renderJsonError($db->getError());
        }
        $this->set('msg', Labels::getLabel('Msg_Successfully_Updated', $this->siteLangId));
        $this->_template->render();
    }

    public function markNotificationRead($notificationId)
    {
        $notificationId = FatUtility::int($notificationId);
        if (1 > $notificationId) {
            $this->renderJsonError(Labels::getLabel('Msg_Invalid_Request', $this->siteLangId));
        }
        $userId = $this->userParentId;
        $srch = Notifications::getSearchObject();
        $srch->addCondition('unt.unotification_user_id', '=', $userId);
        $srch->addCondition('unt.unotification_id', '=', $notificationId);
        $srch->setPageSize(1);
        $rs = $srch->getResultSet();
        $notification = FatApp::getDb()->fetch($rs);
        if (!($notification)) {
            $this->renderJsonError(Labels::getLabel('Msg_Invalid_Request', $this->siteLangId));
        }
        $nObj = new Notifications();
        if (!$nObj->readUserNotification($notificationId, $userId)) {
            $this->renderJsonError($nObj->getError());
        }
        $notificationData = Notifications::getAttributesById($notificationId, ['unotification_data', 'unotification_type']);
        $rfqNotificationTypes = Notifications::getRfqModuleNotificationTypes();
        $resData = json_decode($notificationData['unotification_data'], true);
        if (!empty($notificationData) && (in_array($notificationData['unotification_type'], $rfqNotificationTypes))) {
            switch ($notificationData['unotification_type']) {
                case Notifications::NEW_RFQ_SUBMISSION:
                case Notifications::NEW_RFQ_OFFER_SUBMISSION_BY_BUYER:
                case Notifications::NEW_RFQ_OFFER_UPDATE_BY_BUYER:
                    $action = CommonHelper::generateUrl('requestForQuotes', 'view', [$resData['rfq_id']]);
                    break;
                case Notifications::NEW_RFQ_OFFER_SUBMISSION_BY_SELLER:
                case Notifications::NEW_RFQ_OFFER_UPDATE_BY_SELLER:
                    $action = CommonHelper::generateUrl('requestForQuotes', 'requestView', [$resData['rfq_id']]);
                    break;
                case Notifications::INVOICE_SHARED_BY_SELLER:
                case Notifications::INVOICE_PENDING_REMINDER:
                case Notifications::INVOICE_EXPIRED_REMINDER:
                    $action = CommonHelper::generateUrl('invoices', 'payInvoice', [$resData['order_id']]);
                    break;
                case Notifications::INVOICE_REGENERATE_REQUEST_BY_BUYER:
                    $action = CommonHelper::generateUrl('invoices', 'create', [$resData['order_id']]);
                    break;
                default:
                    $action = '';
            }
            $this->set('action', $action);
        }

        $this->set('msg', Labels::getLabel('Msg_Successfully_Updated', $this->siteLangId));
        //$this->_template->render();
        $this->_template->render(false, false, 'json-success.php', true, false);
    }

    public function changePhoneForm($updatePhnFrm = 0)
    {
        $updatePhnFrm = empty($updatePhnFrm) ? (empty($this->user['user_phone']) ? 1 : 0) : $updatePhnFrm;
        $frm = $this->getPhoneNumberForm();
        if (1 > $updatePhnFrm && !empty($this->user['user_phone'])) {
            $frm->fill(['user_phone' => $this->user['user_phone']]);
            $phnFld = $frm->getField('user_phone');
            $phnFld->setFieldTagAttribute('readonly', 'readonly');
        }
        $this->set('dialCode', $this->user['user_dial_code']);
        $this->set('countryIso', User::getUserMeta($this->user['user_id'], 'user_country_iso'));
        $this->set('frm', $frm);
        $this->set('updatePhnFrm', $updatePhnFrm);
        $this->set('siteLangId', $this->siteLangId);
        $json['html'] = $this->_template->render(false, false, 'account/change-phone-form.php', true, false);
        FatUtility::dieJsonSuccess($json);
    }

    private function sendOtp(int $userId, string $countryIso, string $dialCode, int $phone)
    {
        $userObj = new User($userId);
        $dialCode = !empty($dialCode) & '+' != $dialCode[0] ? '+' . $dialCode : $dialCode;
        $otp = $userObj->prepareUserPhoneOtp($countryIso, $dialCode, $phone);
        if (false == $otp) {
            LibHelper::dieJsonError($userObj->getError());
        }
        $phone = $dialCode . $phone;
        $obj = clone $userObj;
        if (false === $obj->sendOtp($phone, $this->user['user_name'], $otp, $this->siteLangId)) {
            LibHelper::dieJsonError($obj->getError());
        }
        return true;
    }

    public function getOtp($updatePhnFrm = 0)
    {
        $frm = $this->getPhoneNumberForm();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());
        if (false === $post) {
            LibHelper::dieJsonError(current($frm->getValidationErrors()));
        }

        $dialCode = FatApp::getPostedData('user_dial_code', FatUtility::VAR_STRING, '');
        $countryIso = FatApp::getPostedData('user_country_iso', FatUtility::VAR_STRING, '');
        $phoneNumber = FatUtility::int($post['user_phone']);
        if (empty($phoneNumber) || (!empty($phoneNumber) && empty($dialCode))) {
            $message = Labels::getLabel("MSG_INVALID_PHONE_NUMBER_FORMAT", $this->siteLangId);
            LibHelper::dieJsonError($message);
        }
        if (1 > $updatePhnFrm && false === UserAuthentication::validateUserPhone($this->user['user_id'], $phoneNumber)) {
            LibHelper::dieJsonError(Labels::getLabel('MSG_INVALID_PHONE_NUMBER', $this->siteLangId));
        }
        $this->sendOtp($this->user['user_id'], trim($countryIso), trim($dialCode), $phoneNumber);
        $this->set('msg', Labels::getLabel('MSG_OTP_SENT!_PLEASE_CHECK_YOUR_PHONE.', $this->siteLangId));
        if (API_CALL) {
            $this->_template->render();
        }

        $otpFrm = $this->getOtpForm();
        $otpFrm->fill(['user_id' => $this->user['user_id']]);
        $this->set('frm', $otpFrm);
        $json['html'] = $this->_template->render(false, false, 'guest-user/otp-form.php', true, false);
        FatUtility::dieJsonSuccess($json);
    }

    public function validateOtp($updatePhnFrm = 0)
    {
        $updateToDb = (1 > $updatePhnFrm ? 1 : 0);
        $this->validateOtpApi($updateToDb);

        if (0 < $updatePhnFrm) {
            $this->changePhoneForm($updatePhnFrm);
            exit;
        }

        $this->_template->render(false, false, 'json-success.php');
    }

    public function resendOtp()
    {
        $dialCode = FatApp::getPostedData('user_dial_code', FatUtility::VAR_STRING, '');
        $countryIso = FatApp::getPostedData('user_country_iso', FatUtility::VAR_STRING, '');
        $phone = FatApp::getPostedData('user_phone', FatUtility::VAR_INT, 0);
        if (!empty($phone)) {
            $this->sendOtp($this->user['user_id'], $countryIso, $dialCode, $phone);
        } else {
            $userObj = new User($this->user['user_id']);
            if (false == $userObj->resendOtp()) {
                $this->renderJsonError($userObj->getError());
            }
        }
        $this->renderJsonSuccess(Labels::getLabel('MSG_OTP_SENT!_PLEASE_CHECK_YOUR_PHONE.', $this->siteLangId));
    }

    public function pushNotifications()
    {
        $page = FatApp::getPostedData('page', FatUtility::VAR_INT, 1);
        $defaultPageSize = FatApp::getConfig('conf_page_size', FatUtility::VAR_INT, 10);
        $pageSize = FatApp::getPostedData('pagesize', FatUtility::VAR_INT, $defaultPageSize);
        $srch = User::getSearchObject();
        $srch->joinTable(UserAuthentication::DB_TBL_USER_AUTH, 'INNER JOIN', 'ua.uauth_user_id = u.user_id', 'ua');
        $srch->addMultipleFields(['uauth_device_os', 'user_regdate']);
        $srch->addCondition('uauth_user_id', '=', $this->user['user_id']);
        $srch->setPageSize(1);
        $rs = $srch->getResultSet();
        $uData = FatApp::getDb()->fetch($rs);

        $srch = PushNotification::getSearchObject(true);
        $srch->addMultipleFields([
            'pnotification_id',
            'pnotification_title',
            'pnotification_description',
            'pnotification_url',
            'pntu_user_id'
        ]);
        $cond = $srch->addCondition('pnotification_status', '=', PushNotification::STATUS_COMPLETED, 'AND');
        $cond->attachCondition('pnotification_status', '=', PushNotification::STATUS_PROCESSING, 'OR');
        $srch->addCondition('pnotification_user_auth_type', '=', User::AUTH_TYPE_REGISTERED);
        $srch->addCondition('pnotification_added_on', '>=', $uData['user_regdate']);
        $cond = $srch->addCondition('pntu_user_id', 'IS', 'mysql_func_NULL', 'AND', true);
        $cond->attachCondition('pntu_user_id', '=', $this->user['user_id'], 'OR');
        $cond = $srch->addCondition('pnotification_device_os', '=', User::DEVICE_OS_BOTH, 'AND');
        $cond->attachCondition('pnotification_device_os', '=', $uData['uauth_device_os'], 'OR');
        $srch->setPageNumber($page);
        $srch->setPageSize($pageSize);
        $srch->addOrder('pnotification_added_on', 'DESC');
        $srch->addOrder('pnotification_id', 'DESC');
        $rs = $srch->getResultSet();
        $pnotificationsArr = FatApp::getDb()->fetchAll($rs);

        $this->set('pnotifications', $pnotificationsArr);
        $this->set('total_pages', $srch->pages());
        $this->set('total_records', $srch->recordCount());
        $this->_template->render();
    }

    public function loadNotifications()
    {
        $srch = new SearchBase(Notifications::DB_TBL, 'unt');
        $srch->addCondition('unt.unotification_user_id', '=', $this->userParentId);
        $srch->addCondition('unt.unotification_type', 'IN', Notifications::getRfqModuleNotificationTypes());
        $srch->addOrder('unotification_is_read', 'ASC');
        $srch->addOrder('unt.unotification_id', 'DESC');
        $srch->doNotCalculateRecords();
        $srch->setPageSize(50);
        $rs = $srch->getResultSet();
        $records = FatApp::getDb()->fetchAll($rs);
        $this->set('records', $records);
        $this->set('siteLangId', $this->siteLangId);
        $this->_template->render(false, false, 'account/load-notifications.php', false, false);
    }
}
