<?php

class RfqCheckoutController extends MyAppController
{

    private $errMessage;

    public function __construct($action)
    {
        parent::__construct($action);

        if (API_CALL) {
            UserAuthentication::checkLogin();
        }
        if (!UserAuthentication::isUserLogged()) {
            FatApp::redirectUser(CommonHelper::generateUrl());
        }

        if (UserAuthentication::isGuestUserLogged()) {
            $user_is_buyer = User::getAttributesById(UserAuthentication::getLoggedUserId(), 'user_is_buyer');
            if (!$user_is_buyer) {
                $this->errMessage = Labels::getLabel('MSG_Please_login_with_buyer_account', $this->siteLangId);
                Message::addErrorMessage($this->errMessage);
                if (FatUtility::isAjaxCall()) {
                    FatUtility::dieWithError(Message::getHtml());
                }
                FatApp::redirectUser(CommonHelper::generateUrl());
            }
        } else {
            $userObj = new User(UserAuthentication::getLoggedUserId());
            $userInfo = $userObj->getUserInfo(array(), false, false);
            if (empty($userInfo['user_phone']) && empty($userInfo['credential_email'])) {
                if (true == SmsArchive::canSendSms()) {
                    $message = Labels::getLabel('MSG_PLEASE_CONFIGURE_YOUR_EMAIL_OR_PHONE', $this->siteLangId);
                } else {
                    $message = Labels::getLabel('MSG_PLEASE_CONFIGURE_YOUR_EMAIL', $this->siteLangId);
                }
                if (API_CALL) {
                    FatUtility::dieJsonError($message);
                }

                if (FatUtility::isAjaxCall()) {
                    $json['status'] = AppConstants::NO;
                    $json['msg'] = $message;
                    $json['url'] = CommonHelper::generateUrl('GuestUser', 'configureEmail');
                    FatUtility::dieJsonError($json);
                }
                Message::addErrorMessage($message);
                FatApp::redirectUser(CommonHelper::generateUrl('GuestUser', 'configureEmail'));
            }
        }

        $this->set('exculdeMainHeaderDiv', true);
    }

    private function isEligibleForNextStep(&$criteria = array())
    {
        if (empty($criteria)) {
            return true;
        }
        foreach ($criteria as $key => $val) {
            switch ($key) {
                case 'isUserLogged':
                    if (!UserAuthentication::isUserLogged() && !UserAuthentication::isGuestUserLogged()) {
                        $key = false;
                        $this->errMessage = Labels::getLabel('MSG_Your_Session_seems_to_be_expired.', $this->siteLangId);
                        Message::addErrorMessage($this->errMessage);
                        return false;
                    }
                    break;
            }
        }
        return true;
    }

    private function validatePaymentUrl(array $orderInfo): bool
    {
        if (empty($orderInfo)) {
            return false;
        }
        $invoiceValidTill = date('Y-m-d H:i:s', strtotime("+24 Hours", strtotime($orderInfo['invoice_added_on'])));
        //$invoiceValidTill = date('Y-m-d H:i:s', strtotime("+2 minutes", strtotime($orderInfo['invoice_added_on'])));
        if (strtotime($invoiceValidTill) <= strtotime(date('Y-m-d H:i:s'))) {
            return false;
        }
        return true;
    }

    public function index(string $orderId)
    {
        /* [ CHECK FOR VALID PAYMENT URL */
        $orderInfo = $this->getOrderDetails($orderId);
        $rfqStatus = RequestForQuote::getRfqStatus($orderInfo['order_rfq_id']);
        if ($rfqStatus['rfq_status'] == RequestForQuote::REQUEST_QUOTE_VALIDITY) {
            Message::addErrorMessage(Labels::getLabel("LBL_Quote_Expired", $this->siteLangId));
            FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'quotedRequests'));
        }

        if (!$this->validatePaymentUrl($orderInfo)) {
            if (User::isBuyer()) {
                Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Payment_URL_OR_Payment_has_been_Made', $this->siteLangId));
                FatApp::redirectUser(CommonHelper::generateUrl('requestForQuotes', 'quotedRequests'));
            } else {
                Message::addErrorMessage(Labels::getLabel('MSG_Invalid_Payment_URL', $this->siteLangId));
                FatUtility::exitWithErrorCode(404);
            }
        }
        $orderObj = new Orders();
        $orderObj->updateOrderInfo($orderId, ['order_wallet_amount_charge_partial' => 0]);
        /* ] */

        $obj = new Extrapage();
        $this->set('pageData', $obj->getContentByPageType(Extrapage::CHECKOUT_PAGE_RIGHT_BLOCK, $this->siteLangId));
        $this->set('addresses', UserAddress::getUserAddresses(UserAuthentication::getLoggedUserId(), $this->siteLangId));
        $this->set('headerData', $obj->getContentByPageType(Extrapage::CHECKOUT_PAGE_HEADER_BLOCK, $this->siteLangId));
        $this->set('orderId', $orderId);
        $this->_template->render(true, false);
    }

    public function paymentSummary()
    {
        $orderId = FatApp::getPostedData('order_id', FatUtility::VAR_STRING, '');
        $userId = UserAuthentication::getLoggedUserId();
        if ($orderId == '') {
            FatUtility::dieWithError(Labels::getLabel('MSG_Invaild_Request', $this->siteLangId));
        }

        $criteria = array('isUserLogged' => true);
        if (!$this->isEligibleForNextStep($criteria)) {
            $this->errMessage = Labels::getLabel('MSG_Something_went_wrong,_please_try_after_some_time.', $this->siteLangId);
            if (API_CALL) {
                LibHelper::dieJsonError($this->errMessage);
            }
            if (Message::getErrorCount()) {
                $this->errMessage = Message::getHtml();
            }
            FatUtility::dieWithError($this->errMessage);
        }
        $orderInfo = $this->getOrderDetails($orderId);
        $shopCountryId = $orderInfo['shop_country_id'];
        $cartSummary = $orderInfo['cart_summary'];
        //echo '<pre>'; print_r($cartSummary); echo '</pre>'; exit;
        unset($orderInfo['cart_summary']);

        /* Payment Methods[ */
        $pmSrch = PaymentMethods::getSearchObject($this->siteLangId);
        $pmSrch->doNotCalculateRecords();
        $pmSrch->doNotLimitRecords();
        $pmSrch->addMultipleFields(array('pmethod_id', 'IFNULL(pmethod_name, pmethod_identifier) as pmethod_name', 'pmethod_code', 'pmethod_description'));

        if (!$cartSummary["isCodEnabled"]) {
            $pmSrch->addCondition('pmethod_code', '!=', 'CashOnDelivery');
        }
        $localCountries = Countries::getLocalCountriesArr();
        /* $localPaymentGateways = PaymentMethods::getLocalPaymentGateways(); */
        if (in_array($shopCountryId, $localCountries)) {
            /* $pmSrch->addCondition('pmethod_id', 'IN', $localPaymentGateways); */
            $cnd = $pmSrch->addCondition('pmethod_type', '=', PaymentMethods::PAYMENT_METHOD_TYPE_LOCAL, 'AND');
            $cnd->attachCondition('pmethod_type', '=', PaymentMethods::PAYMENT_METHOD_TYPE_ALL);
        } else {
            /* $pmSrch->addCondition('pmethod_id', '!=', PaymentMethods::PAYMENT_METHOD_PAYULATAM); */
            $cnd = $pmSrch->addCondition('pmethod_type', '=', PaymentMethods::PAYMENT_METHOD_TYPE_GLOBAL, 'AND');
            $cnd->attachCondition('pmethod_type', '=', PaymentMethods::PAYMENT_METHOD_TYPE_ALL);
        }

        $pmRs = $pmSrch->getResultSet();
        $paymentMethods = FatApp::getDb()->fetchAll($pmRs);
        /* ] */
        $userWalletBalance = User::getUserBalance($userId, true);
        if (false === API_CALL) {
            $WalletPaymentForm = $this->getWalletPaymentForm($this->siteLangId);
            $confirmForm = $this->getConfirmFormWithNoAmount();

            if ((FatUtility::convertToType($userWalletBalance, FatUtility::VAR_FLOAT) > 0) && $cartSummary['cartWalletSelected'] && $cartSummary['orderNetAmount'] > 0) {
                $WalletPaymentForm->addFormTagAttribute('action', CommonHelper::generateUrl('WalletPay', 'Charge', array($orderId)));
                $WalletPaymentForm->fill(array('order_id' => $orderId));
                $WalletPaymentForm->setFormTagAttribute('onsubmit', 'confirmOrder(this); return(false);');
                $WalletPaymentForm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Pay_Now', $this->siteLangId));
            }

            if ($cartSummary['orderNetAmount'] <= 0) {
                $confirmForm->addFormTagAttribute('action', CommonHelper::generateUrl('ConfirmPay', 'Charge', array($orderId)));
                $confirmForm->fill(array('order_id' => $orderId));
                $confirmForm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Confirm_Order', $this->siteLangId));
            }
        }

        $this->set('paymentMethods', $paymentMethods);
        $this->set('userWalletBalance', $userWalletBalance);
        $this->set('cartSummary', $cartSummary);
        if (false === API_CALL) {
            $this->set('excludePaymentGatewaysArr', AppConstants::getExcludePaymentGatewayArr());
            $this->set('orderInfo', $orderInfo);
            $this->set('confirmForm', $confirmForm);
            $this->set('WalletPaymentForm', $WalletPaymentForm);
        }

        if (API_CALL) {
            $this->set('orderType', $orderInfo['order_type']);
            $this->_template->render();
        }
        $this->_template->render(false, false);
    }

    private function getOrderDetails(string $orderId): array
    {
        $srch = new OrderProductSearch($this->siteLangId, true, true);
        $srch->joinTable(Invoice::DB_TBL, 'INNER JOIN', 'invoice.invoice_order_id = op.op_order_id', 'invoice');
        $srch->joinTable(Shop::DB_TBL, 'INNER JOIN', 'shop.shop_id = op.op_shop_id', 'shop');
        $srch->joinSellerProducts();
        $srch->joinOrderUser();
        $srch->addOrderProductCharges();
        $srch->addStatusCondition(unserialize(FatApp::getConfig("CONF_VENDOR_ORDER_STATUS")));
        $srch->addCondition('order_user_id', '=', UserAuthentication::getLoggedUserId());
        $srch->addCondition('op_order_id', '=', $orderId);
        $srch->addCondition('order_is_rfq', '=', AppConstants::YES);
        $srch->addCondition('order_is_paid', 'IN', Orders::getUnpaidStatus());
        $srch->addCondition('op_status_id', '!=', FatApp::getConfig('CONF_DEFAULT_CANCEL_ORDER_STATUS'));

        $rs = $srch->getResultSet();
        $orderDetail = FatApp::getDb()->fetch($rs);
        if (empty($orderDetail)) {
            return [];
        }
        $isWalletSelected = 0;
        $walletCharges = 0;
        if ($orderDetail['order_wallet_amount_charge_partial'] > 0) {
            $isWalletSelected = 1;
            $walletCharges = $orderDetail['order_wallet_amount_charge_partial'];
        }


        $orderObj = new Orders();
        $cartSummary = array(
            'cartTotal' => $orderDetail['order_net_amount'] - $orderDetail['op_actual_shipping_charges'] - $orderDetail['order_tax_charged'],
            'shippingTotal' => $orderDetail['op_actual_shipping_charges'],
            'originalShipping' => $orderDetail['op_actual_shipping_charges'],
            'cartTaxTotal' => $orderDetail['order_tax_charged'],
            'cartDiscounts' => 0,
            'cartVolumeDiscount' => 0,
            'cartRewardPoints' => 0,
            'cartWalletSelected' => $isWalletSelected,
            'siteCommission' => $orderDetail['order_site_commission'],
            'orderNetAmount' => $orderDetail['order_net_amount'],
            'WalletAmountCharge' => $walletCharges,
            'isCodEnabled' => 0,
            'isCodValidForNetAmt' => 0,
            'orderPaymentGatewayCharges' => $orderDetail['order_partial_payment'] - $walletCharges,
            'netChargeAmount' => $orderDetail['order_partial_payment'] - $walletCharges,
            'taxOptions' => $orderDetail['op_product_tax_options'],
            'prodTaxOptions' => $orderDetail['op_product_tax_options'],
            'total_paid_amount' => $orderObj->getOrderPaymentPaid($orderId),
            'total_amount' => $orderDetail['order_partial_payment'],
        );
        $orderDetail['cart_summary'] = $cartSummary;
        return $orderDetail;
    }

    private function getPaymentMethodData($pmethod_id)
    {
        $pmSrch = PaymentMethods::getSearchObject($this->siteLangId);
        $pmSrch->doNotCalculateRecords();
        $pmSrch->doNotLimitRecords();
        $pmSrch->addMultipleFields(array('pmethod_id', 'IFNULL(pmethod_name, pmethod_identifier) as pmethod_name', 'pmethod_code', 'pmethod_description'));
        $pmSrch->addCondition('pmethod_id', '=', $pmethod_id);
        $pmRs = $pmSrch->getResultSet();
        $paymentMethod = FatApp::getDb()->fetch($pmRs);

        if (!$paymentMethod) {
            FatUtility::dieWithError(Labels::getLabel("MSG_Selected_Payment_method_not_found!", $this->siteLangId));
        }
        return $paymentMethod;
    }

    public function paymentTab($order_id, $pmethod_id)
    {
        $pmethod_id = FatUtility::int($pmethod_id);
        if (!$pmethod_id) {
            FatUtility::dieWithError(Labels::getLabel("MSG_Invalid_Request!", $this->siteLangId));
        }

        if (!UserAuthentication::isUserLogged() && !UserAuthentication::isGuestUserLogged()) {
            /* Message::addErrorMessage( Labels::getLabel('MSG_Your_Session_seems_to_be_expired.', $this->siteLangId) );
              FatUtility::dieWithError( Message::getHtml() ); */
            FatUtility::dieWithError(Labels::getLabel('MSG_Your_Session_seems_to_be_expired.', $this->siteLangId));
        }

        $srch = Orders::getSearchObject();
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addCondition('order_id', '=', $order_id);
        $srch->addCondition('order_is_paid', 'IN', Orders::getUnpaidStatus());
        $rs = $srch->getResultSet();
        $orderInfo = FatApp::getDb()->fetch($rs);
        /* $orderObj = new Orders();
          $orderInfo = $orderObj->getOrderById( $order_id, $this->siteLangId, array('payment_status' => 0) ); */
        if (!$orderInfo) {
            /* Message::addErrorMessage( Labels::getLabel('MSG_INVALID_ORDER_PAID_CANCELLED', $this->siteLangId) );
              $this->set('error', Message::getHtml() ); */
            FatUtility::dieWithError(Labels::getLabel('MSG_INVALID_ORDER_PAID_CANCELLED', $this->siteLangId));
        }

        //commonHelper::printArray($orderInfo);

        $paymentMethod = $this->getPaymentMethodData($pmethod_id);
        $frm = $this->getPaymentTabForm($paymentMethod['pmethod_code']);
        $controller = $paymentMethod['pmethod_code'] . 'Pay';
        $frm->setFormTagAttribute('action', CommonHelper::generateUrl($controller, 'charge', array($order_id)));
        $frm->fill(
                array(
                    'order_type' => $orderInfo['order_type'],
                    'order_id' => $order_id,
                    'pmethod_id' => $pmethod_id
                )
        );

        $this->set('orderInfo', $orderInfo);
        $this->set('paymentMethod', $paymentMethod);
        $this->set('frm', $frm);
        $this->_template->render(false, false, '', false, false);
    }

    public function ConfirmOrder()
    {
        $pmethod_id = FatApp::getPostedData('pmethod_id', FatUtility::VAR_INT, 0);
        $order_id = FatApp::getPostedData("order_id", FatUtility::VAR_STRING, "");
        $user_id = UserAuthentication::getLoggedUserId();
        $orderDetails = $this->getOrderDetails($order_id);
        $cartSummary = $orderDetails['cart_summary'];
        $userWalletBalance = FatUtility::convertToType(User::getUserBalance($user_id, true), FatUtility::VAR_FLOAT);
        $orderNetAmount = isset($cartSummary['orderPaymentGatewayCharges']) ? FatUtility::convertToType($cartSummary['orderPaymentGatewayCharges'], FatUtility::VAR_FLOAT) : 0;

        if (0 < $pmethod_id) {
            $paymentMethodRow = PaymentMethods::getAttributesById($pmethod_id);
            if (!$paymentMethodRow || $paymentMethodRow['pmethod_active'] != AppConstants::ACTIVE) {
                $this->errMessage = Labels::getLabel("LBL_Invalid_Payment_method,_Please_contact_Webadmin.", $this->siteLangId);
                if (API_CALL) {
                    LibHelper::dieJsonError($this->errMessage);
                }
                Message::addErrorMessage($this->errMessage);
                FatUtility::dieWithError(Message::getHtml());
            }
        }

        /* [ CHECK FOR VALID PAYMENT URL */
        if (!$this->validatePaymentUrl($orderDetails)) {
            $this->errMessage = Labels::getLabel('MSG_Invalid_Payment_Url.', $this->siteLangId);
            if (API_CALL) {
                LibHelper::dieJsonError($this->errMessage);
            }
            if (Message::getErrorCount()) {
                $this->errMessage = Message::getHtml();
            }
            FatUtility::dieWithError($this->errMessage);
        }
        /* ] */


        if (!empty($paymentMethodRow) && strtolower($paymentMethodRow['pmethod_code']) == "cashondelivery" && $cartSummary['cartWalletSelected'] && $userWalletBalance < $orderNetAmount) {
            $str = Labels::getLabel('MSG_Wallet_can_not_be_used_along_with_{COD}', $this->siteLangId);
            $str = str_replace('{cod}', $paymentMethodRow['pmethod_identifier'], $str);
            if (API_CALL) {
                LibHelper::dieJsonError($str);
            }
            FatUtility::dieWithError($str);
        }

        if (API_CALL) {
            $paymentUrl = '';
            $sendToWeb = 1;
            if (0 < $pmethod_id) {
                $controller = $paymentMethodRow['pmethod_code'] . 'Pay';
                $paymentUrl = CommonHelper::generateFullUrl($controller, 'charge', array($order_id));
            }
            if ($cartSummary['cartWalletSelected'] && $userWalletBalance >= $orderNetAmount) {
                $sendToWeb = $pmethod_id = 0;
                $paymentUrl = CommonHelper::generateFullUrl('WalletPay', 'charge', array($order_id));
            }
            if (empty($paymentUrl)) {
                LibHelper::dieJsonError(Labels::getLabel('MSG_Please_Select_Payment_Method', $this->siteLangId));
            }
            $this->set('sendToWeb', $sendToWeb);
            $this->set('orderPayment', $paymentUrl);
        }

        /* ConfirmOrder function is called for both wallet payments and for paymentgateway selection as well. */
        $criteria = array('isUserLogged' => true);

        if (!$this->isEligibleForNextStep($criteria)) {
            $this->errMessage = Labels::getLabel('MSG_Something_went_wrong,_please_try_after_some_time.', $this->siteLangId);
            if (API_CALL) {
                FatUtility::dieJsonError($this->errMessage);
            }
            if (Message::getErrorCount()) {
                $this->errMessage = Message::getHtml();
            }
            FatUtility::dieWithError($this->errMessage);
        }

        if ($cartSummary['cartWalletSelected'] && $userWalletBalance >= $orderNetAmount && !$pmethod_id) {
            if (API_CALL) {
                $this->_template->render();
            }
            $this->_template->render(false, false, 'json-success.php');
            exit;
        }

        $post = FatApp::getPostedData();
        // commonHelper::printArray($post); die;
        $paymentMethodRow = PaymentMethods::getAttributesById($pmethod_id);
        if (!$paymentMethodRow || $paymentMethodRow['pmethod_active'] != AppConstants::ACTIVE) {
            $this->errMessage = Labels::getLabel("LBL_Invalid_Payment_method,_Please_contact_Webadmin.", $this->siteLangId);
            if (API_CALL) {
                LibHelper::dieJsonError($this->errMessage);
            }
            Message::addErrorMessage($this->errMessage);
            FatUtility::dieWithError(Message::getHtml());
        }

        if (false === API_CALL && strtolower($paymentMethodRow['pmethod_code']) == 'cashondelivery' && FatApp::getConfig('CONF_RECAPTCHA_SITEKEY', FatUtility::VAR_STRING, '' && FatApp::getConfig('CONF_RECAPTCHA_SECRETKEY', FatUtility::VAR_STRING, '') != '')) {
            if (!CommonHelper::verifyCaptcha()) {
                Message::addErrorMessage(Labels::getLabel('MSG_That_captcha_was_incorrect', $this->siteLangId));
                FatUtility::dieWithError(Message::getHtml());
                //FatApp::redirectUser(CommonHelper::generateUrl('Custom', 'ContactUs'));
            }
        }

        /* if ($userWalletBalance >= $cartSummary['orderPaymentGatewayCharges'] && $cartSummary['cartWalletSelected'] && !$pmethod_id) {
          $frm = $this->getWalletPaymentForm($this->siteLangId);
          } else {
          $frm = $this->getPaymentTabForm();
          } */
        $frm = $this->getPaymentTabForm();


        $post = $frm->getFormDataFromArray($post);
        if (!isset($post['order_id']) || $post['order_id'] == '') {
            $this->errMessage = Labels::getLabel('MSG_Invalid_Request', $this->siteLangId);
            if (API_CALL) {
                LibHelper::dieJsonError($this->errMessage);
            }
            Message::addErrorMessage($this->errMessage);
            FatUtility::dieWithError(Message::getHtml());
        }

        $orderObj = new Orders();
        $srch = Orders::getSearchObject();
        $srch->doNotCalculateRecords();
        $srch->doNotLimitRecords();
        $srch->addCondition('order_id', '=', $order_id);
        $srch->addCondition('order_user_id', '=', $user_id);
        $srch->addCondition('order_is_paid', 'IN', Orders::getUnpaidStatus());
        $rs = $srch->getResultSet();
        $orderInfo = FatApp::getDb()->fetch($rs);

        if (!$orderInfo) {
            $this->errMessage = Labels::getLabel('MSG_INVALID_ORDER_PAID_CANCELLED', $this->siteLangId);
            if (API_CALL) {
                LibHelper::dieJsonError($this->errMessage);
            }
            Message::addErrorMessage($this->errMessage);
            FatUtility::dieWithError(Message::getHtml());
        }
        if ($cartSummary['cartWalletSelected'] && $cartSummary['orderPaymentGatewayCharges'] == 0) {
            $this->errMessage = Labels::getLabel('MSG_Try_to_pay_using_wallet_balance_as_amount_for_payment_gateway_is_not_enough.', $this->siteLangId);
            if (API_CALL) {
                LibHelper::dieJsonError($this->errMessage);
            }
            Message::addErrorMessage($this->errMessage);
            FatUtility::dieWithError(Message::getHtml());
        }

        if ($cartSummary['orderPaymentGatewayCharges'] == 0 && $pmethod_id) {
            $this->errMessage = Labels::getLabel('MSG_Amount_for_payment_gateway_must_be_greater_than_zero.', $this->siteLangId);
            if (API_CALL) {
                LibHelper::dieJsonError($this->errMessage);
            }
            Message::addErrorMessage($this->errMessage);
            FatUtility::dieWithError(Message::getHtml());
        }

        if ($pmethod_id) {
            $orderObj->updateOrderInfo($order_id, ['order_pmethod_id' => $pmethod_id]);
        }

        if (API_CALL) {
            $this->_template->render();
        }

        $this->_template->render(false, false, 'json-success.php');
    }

    private function getPaymentTabForm($paymentMethodCode = '')
    {
        $frm = new Form('frmPaymentTabForm');
        $frm->setFormTagAttribute('id', 'frmPaymentTabForm');

        if (strtolower($paymentMethodCode) == "cashondelivery") {
            CommonHelper::addCaptchaField($frm);
        }

        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Confirm_Payment', $this->siteLangId));
        $frm->addHiddenField('', 'order_type');
        $frm->addHiddenField('', 'order_id');
        $frm->addHiddenField('', 'pmethod_id');
        return $frm;
    }

    private function getConfirmFormWithNoAmount()
    {
        $frm = new Form('frmConfirmForm');
        $frm->addHiddenField('', 'order_id');
        return $frm;
    }

    public function getFinancialSummary()
    {
        $orderId = FatApp::getPostedData('order_id', FatUtility::VAR_STRING, '');
        $orderDetails = $this->getOrderDetails($orderId);
        $cartSummary = $orderDetails['cart_summary'];
        unset($orderDetails['cart_summary']);
        $orderObj = new Orders();
        $address = $orderObj->getOrderAddresses($orderId);
        $this->set('product', $orderDetails);
        $this->set('cartSummary', $cartSummary);
        $this->set('defaultAddress', $address[Orders::SHIPPING_ADDRESS_TYPE]);
        $this->set('hasPhysicalProd', ($orderDetails['op_product_type'] == Product::PRODUCT_TYPE_PHYSICAL) ? true : false);
        $this->_template->render(false, false);
    }

    private function getWalletPaymentForm($langId)
    {
        $frm = new Form('frmWalletPayment');
        $frm->addHiddenField('', 'order_id');
        return $frm;
    }

    public function walletSelection()
    {
        $orderId = FatApp::getPostedData('order_id', FatUtility::VAR_STRING, '');
        $isWalletSelected = FatApp::getPostedData('payFromWallet', FatUtility::VAR_INT, 0);
        $srch = Orders::getSearchObject();
        $srch->addCondition('order_id', '=', $orderId);
        $srch->addCondition('order_is_rfq', '=', AppConstants::YES);
        $srch->addFld('order_partial_payment');
        $rs = $srch->getResultSet();
        $row = FatApp::getDb()->fetch($rs);
        if ($row == false) {
            // need to update    
        }

        $amountToPay = $row['order_partial_payment'];
        $userBalance = FatUtility::convertToType(User::getUserBalance(UserAuthentication::getLoggedUserId(), true), FatUtility::VAR_FLOAT);
        $walletCharges = ($isWalletSelected) ? min($userBalance, $amountToPay) : 0;
        $orderObj = new Orders();
        $orderObj->updateOrderInfo($orderId, ['order_is_wallet_selected' => $isWalletSelected, 'order_wallet_amount_charge_partial' => $walletCharges]);
        $this->_template->render(false, false, 'json-success.php');
    }

}
